/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/* Original source keras/models.py */
import { dispose, io, serialization, util } from '@tensorflow/tfjs-core';
import { getUid } from './backend/state';
import { Input } from './engine/input_layer';
import { getSourceInputs, Node } from './engine/topology';
import { LayersModel } from './engine/training';
import { NotImplementedError, RuntimeError, ValueError } from './errors';
import { deserialize } from './layers/serialization';
import * as generic_utils from './utils/generic_utils';
import { convertPythonicToTs } from './utils/serialization_utils';
import { getExactlyOneShape } from './utils/types_utils';
/**
 * Parses a JSON model configuration file and returns a model instance.
 *
 * ```js
 * // This example shows how to serialize a model using `toJSON()` and
 * // deserialize it as another model using `tf.models.modelFromJSON()`.
 * // Note: this example serializes and deserializes only the topology
 * // of the model; the weights of the loaded model will be different
 * // from those of the the original model, due to random weight
 * // initialization.
 * // To load the topology and weights of a model, use `tf.loadLayersModel()`.
 * const model1 = tf.sequential();
 * model1.add(tf.layers.repeatVector({inputShape: [2], n: 4}));
 * // Serialize `model1` as a JSON object.
 * const model1JSON = model1.toJSON(null, false);
 * model1.summary();
 *
 * const model2 = await tf.models.modelFromJSON(model1JSON);
 * model2.summary();
 * ```
 *
 *  @param modelAndWeightsConfig JSON object or string encoding a model and
 *       weights configuration. It can also be only the topology JSON of the
 *       model, in which case the weights will not be loaded.
 *  @param custom_objects Optional dictionary mapping names
 *       (strings) to custom classes or functions to be
 *       considered during deserialization.
 * @returns A TensorFlow.js Layers `tf.LayersModel` instance (uncompiled).
 */
export async function modelFromJSON(modelAndWeightsConfig, customObjects) {
    if (!('modelTopology' in modelAndWeightsConfig)) {
        modelAndWeightsConfig = { modelTopology: modelAndWeightsConfig };
    }
    modelAndWeightsConfig = modelAndWeightsConfig;
    let modelTopology = modelAndWeightsConfig.modelTopology;
    if (modelTopology['model_config'] != null) {
        // If the model-topology JSON contains a 'model_config' field, then it is
        // a full model JSON (e.g., from `keras.Model.save()`), which contains
        // not only the model's architecture in its 'model_config' field, but
        // additional information such as the model's optimizer. We use only the
        // 'model_config' field currently.
        modelTopology = modelTopology['model_config'];
    }
    const tsConfig = convertPythonicToTs(modelTopology);
    const model = deserialize(tsConfig, customObjects);
    if (modelAndWeightsConfig.weightsManifest != null) {
        // Load the weight values keyed by the original tensor names in the model
        // file that was loaded.  These should match the keys of the weight
        // manifest.
        const weightValues = await io.loadWeights(modelAndWeightsConfig.weightsManifest, modelAndWeightsConfig.pathPrefix, model.weights.map(weight => weight.originalName));
        // Map the weights to the unique tensor names generated during model loading
        const uniqueWeightValues = {};
        for (const weight of model.weights) {
            uniqueWeightValues[weight.originalName] =
                weightValues[weight.originalName];
        }
        model.loadWeights(uniqueWeightValues);
        // Dispose temporary weight values.
        dispose(weightValues);
    }
    return model;
}
/**
 * Load a model, including its topology and optionally weights.  See the
 * Tutorial named "How to import a Keras Model" for usage examples.
 *
 * Example 1: Save `model`'s topology and weights to browser [local
 * storage](https://developer.mozilla.org/en-US/docs/Web/API/Window/localStorage);
 * then load it back.
 *
 * ```js
 * const model = tf.sequential(
 *     {layers: [tf.layers.dense({units: 1, inputShape: [3]})]});
 * console.log('Prediction from original model:');
 * model.predict(tf.ones([1, 3])).print();
 *
 * const saveResults = await model.save('localstorage://my-model-1');
 *
 * const loadedModel = await tf.loadLayersModel('localstorage://my-model-1');
 * console.log('Prediction from loaded model:');
 * loadedModel.predict(tf.ones([1, 3])).print();
 * ```
 *
 * Example 2. Saving `model`'s topology and weights to browser
 * [IndexedDB](https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API);
 * then load it back.
 *
 * ```js
 * const model = tf.sequential(
 *     {layers: [tf.layers.dense({units: 1, inputShape: [3]})]});
 * console.log('Prediction from original model:');
 * model.predict(tf.ones([1, 3])).print();
 *
 * const saveResults = await model.save('indexeddb://my-model-1');
 *
 * const loadedModel = await tf.loadLayersModel('indexeddb://my-model-1');
 * console.log('Prediction from loaded model:');
 * loadedModel.predict(tf.ones([1, 3])).print();
 * ```
 *
 * Example 3. Load a model from user-selected files from HTML
 * [file input
 * elements](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input/file).
 *
 * ```js
 * // Note: this code snippet will not work without the HTML elements in the
 * //   page
 * const jsonUpload = document.getElementById('json-upload');
 * const weightsUpload = document.getElementById('weights-upload');
 *
 * const model = await tf.loadLayersModel(
 *     tf.io.browserFiles([jsonUpload.files[0], weightsUpload.files[0]]));
 * ```
 *
 * Example 4. Load a model from an HTTP server.
 *
 * ```js
 * const model = await
 *     tf.loadLayersModel('https://storage.googleapis.com/tfjs-models/tfjs/iris_v1/model.json');
 * model.summary();
 * ```
 *
 * @param pathOrIOHandler Can be either of the two formats
 *   1. A string path to the `ModelAndWeightsConfig` JSON describing
 *      the model in the canonical TensorFlow.js format. This path will be
 *      interpreted as a relative HTTP path, to which `fetch` will be used to
 *      request the model topology and weight manifest JSON.
 *      The content of the JSON file is assumed to be a JSON object with the
 *      following fields and values:
 *      - 'modelTopology': A JSON object that can be either of:
 *        1. a model architecture JSON consistent with the format of the return
 *            value of `keras.Model.to_json()`
 *        2. a full model JSON in the format of `keras.models.save_model()`.
 *      - 'weightsManifest': A TensorFlow.js weights manifest.
 *      See the Python converter function `save_model()` for more details.
 *      It is also assumed that model weights can be accessed from relative
 *      paths described by the `paths` fields in weights manifest.
 *   2. An `tf.io.IOHandler` object that loads model artifacts with its `load`
 *      method.
 * @param options Optional configuration arguments for the model loading,
 *   including:
 *   - `strict`: Require that the provided weights exactly match those required
 *     by the layers.  Default true.  Passing false means that both extra
 *     weights and missing weights will be silently ignored.
 *   - `onProgress`: A progress callback of the form:
 *     `(fraction: number) => void`. This callback can be used to monitor the
 *     model-loading process.
 * @returns A `Promise` of `tf.LayersModel`, with the topology and weights
 *     loaded.
 */
export async function loadLayersModelInternal(pathOrIOHandler, options) {
    if (options == null) {
        options = {};
    }
    if (typeof pathOrIOHandler === 'string') {
        const handlers = io.getLoadHandlers(pathOrIOHandler, options);
        if (handlers.length === 0) {
            // For backward compatibility: if no load handler can be found,
            // assume it is a relative http path.
            // TODO(cais): Reformat the args into a single `LoadOptions` once the core
            // is refactored.
            handlers.push(io.browserHTTPRequest(pathOrIOHandler, options));
        }
        else if (handlers.length > 1) {
            throw new ValueError(`Found more than one (${handlers.length}) load handlers for ` +
                `URL '${pathOrIOHandler}'`);
        }
        pathOrIOHandler = handlers[0];
    }
    return loadLayersModelFromIOHandler(pathOrIOHandler, undefined, options);
}
/**
 * Load a model and optionally its weights, using an IOHandler object.
 *
 * @param handler The instance of `IOHandler` to be used during the model
 *   loading.
 * @param customObjects Any optional custom objects to be used during model
 *   loading.
 * @param strict Whether the weight loading will be done in strict mode.
 *   Default: `true`.
 */
export async function loadLayersModelFromIOHandler(handler, customObjects, options) {
    if (options == null) {
        options = {};
    }
    if (handler.load == null) {
        throw new ValueError('Cannot proceed with model loading because the IOHandler provided ' +
            'does not have the `load` method implemented.');
    }
    const artifacts = await handler.load();
    let modelTopology = artifacts.modelTopology;
    if (modelTopology['model_config'] != null) {
        modelTopology = modelTopology['model_config'];
    }
    const strict = options.strict == null ? true : options.strict;
    // If weights are provided and the weight-loading mode is strict, use
    // fast weight initialization. This skips costly initializers such as
    // 'orthogonal' and saves unnecessary computation in cases where
    // the initialized weight values will immediately be overwritten by
    // loaded weight values.
    const fastWeightInit = artifacts.weightData != null && artifacts.weightSpecs != null && strict;
    const model = deserialize(convertPythonicToTs(modelTopology), customObjects, fastWeightInit);
    const trainingConfig = artifacts.trainingConfig;
    if (trainingConfig != null) {
        model.loadTrainingConfig(trainingConfig);
    }
    if (artifacts.userDefinedMetadata != null) {
        model.setUserDefinedMetadata(artifacts.userDefinedMetadata);
    }
    // If weightData is present, load the weights into the model.
    if (artifacts.weightData != null) {
        // Loading weights requires weightSpecs.
        if (artifacts.weightSpecs == null) {
            throw new ValueError('LayersModel artifacts contains weight data, but not weight specs. ' +
                'Therefore loading of weights cannot proceed.');
        }
        const { modelWeights, optimizerWeights } = decodeModelAndOptimizerWeights(artifacts.weightData, artifacts.weightSpecs);
        model.loadWeights(modelWeights, strict);
        if (model.optimizer != null && optimizerWeights.length > 0) {
            await model.optimizer.setWeights(optimizerWeights);
        }
        // Dispose temporary weight values.
        dispose(modelWeights);
        dispose(optimizerWeights.map(w => w.tensor));
    }
    return model;
}
function decodeModelAndOptimizerWeights(buffer, specs) {
    const name2Tensor = io.decodeWeights(buffer, specs);
    const modelWeights = {};
    const optimizerWeights = [];
    specs.forEach(spec => {
        if (spec.group === 'optimizer') {
            optimizerWeights.push({ name: spec.name, tensor: name2Tensor[spec.name] });
        }
        else {
            modelWeights[spec.name] = name2Tensor[spec.name];
        }
    });
    return { modelWeights, optimizerWeights };
}
/**
 * A model with a stack of layers, feeding linearly from one to the next.
 *
 * `tf.sequential` is a factory function that creates an instance of
 * `tf.Sequential`.
 *
 * ```js
 *  // Define a model for linear regression.
 *  const model = tf.sequential();
 *  model.add(tf.layers.dense({units: 1, inputShape: [1]}));
 *
 *  // Prepare the model for training: Specify the loss and the optimizer.
 *  model.compile({loss: 'meanSquaredError', optimizer: 'sgd'});
 *
 *  // Generate some synthetic data for training.
 *  const xs = tf.tensor2d([1, 2, 3, 4], [4, 1]);
 *  const ys = tf.tensor2d([1, 3, 5, 7], [4, 1]);
 *
 *  // Train the model using the data then do inference on a data point the
 *  // model hasn't seen:
 *  await model.fit(xs, ys);
 *  model.predict(tf.tensor2d([5], [1, 1])).print();
 * ```
 *
 * @doc {heading: 'Models', subheading: 'Classes'}
 */
export class Sequential extends LayersModel {
    constructor(args) {
        super({ inputs: [], outputs: [] });
        args = args || {};
        this.trainable = true;
        this.built = false;
        // Set model name.
        this.name = (args.name != null) ? args.name : getUid('sequential_');
        // Add to the model any layers passed to the constructor.
        if (args.layers != null) {
            for (const layer of args.layers) {
                this.add(layer);
            }
        }
    }
    // Helper function to Sequential.add  Throws if the new output shape will be
    // invalid.
    checkShape(layer) {
        const shape = layer.inboundNodes[0].outputTensors[0].shape;
        if (shape.some(x => x < 0)) {
            throw new ValueError('Negative dimension size caused by adding layer ' +
                `${layer.name} with input shape [` +
                `${layer.inboundNodes[0].inputTensors[0].shape}]`);
        }
    }
    /**
     * Adds a layer instance on top of the layer stack.
     *
     * ```js
     *  const model = tf.sequential();
     *  model.add(tf.layers.dense({units: 8, inputShape: [1]}));
     *  model.add(tf.layers.dense({units: 4, activation: 'relu6'}));
     *  model.add(tf.layers.dense({units: 1, activation: 'relu6'}));
     *  // Note that the untrained model is random at this point.
     *  model.predict(tf.randomNormal([10, 1])).print();
     * ```
     * @param layer Layer instance.
     *
     * @exception ValueError In case the `layer` argument does not know its
     * input shape.
     * @exception ValueError In case the `layer` argument has multiple output
     *   tensors, or is already connected somewhere else (forbidden in
     *   `Sequential` models).
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    add(layer) {
        const isLayerModelInstance = layer instanceof Sequential || layer instanceof LayersModel;
        let modelLayer;
        if (isLayerModelInstance) {
            modelLayer = layer;
            if (modelLayer.outputs.length !== 1) {
                throw new ValueError('All layers in a Sequential model ' +
                    'should have a single output tensor. ' +
                    'For multi-output layers, ' +
                    'use the functional API.');
            }
            if (modelLayer.inputs.length !== 1) {
                throw new ValueError('All layers in a Sequential model ' +
                    'should have a single input tensor. ' +
                    'For multi-input layers, ' +
                    'use the functional API.');
            }
        }
        if (this.outputs.length === 0) {
            // first layer in model: check that it is an input layer
            if (layer.inboundNodes.length === 0) {
                // create an input layer
                if (layer.batchInputShape == null) {
                    throw new ValueError('The first layer in a Sequential model must ' +
                        'get an `inputShape` or `batchInputShape` argument.');
                }
                // Instantiate the input layer.
                const x = Input({
                    batchShape: layer.batchInputShape,
                    dtype: layer.dtype,
                    name: layer.name + '_input'
                });
                // This will build the current layer and create the node connecting
                // the current layer to the input layer we just created.
                layer.apply(x);
            }
            if (isLayerModelInstance) {
                this.outputs = modelLayer.outputs;
                this.inputs = modelLayer.inputs;
            }
            else {
                if (layer.inboundNodes.length !== 1) {
                    throw new ValueError('A layer added to a Sequential model must not already be ' +
                        `connected somewhere else. LayersModel received layer ${layer.name} ` +
                        `which has ${layer.inboundNodes.length} pre-existing inbound ` +
                        'connections.');
                }
                if (layer.inboundNodes[0].outputTensors.length !== 1) {
                    throw new ValueError('All layers in a Sequential model ' +
                        'should have a single output tensor. ' +
                        'For multi-output layers, ' +
                        'use the functional API.');
                }
                this.checkShape(layer);
                this.outputs = [layer.inboundNodes[0].outputTensors[0]];
                this.inputs = getSourceInputs(this.outputs[0]);
            }
            this.inboundNodes = [];
            // We create an input node, which we will keep updated
            // as we add more layers.
            // (This call has side effects.)
            // tslint:disable-next-line:no-unused-expression
            new Node({
                outboundLayer: this,
                inboundLayers: [],
                nodeIndices: [],
                tensorIndices: [],
                inputTensors: this.inputs,
                outputTensors: this.outputs,
                // no model-level masking for now
                inputMasks: generic_utils.pyListRepeat(null, this.inputs.length),
                outputMasks: [null],
                inputShapes: this.inputs.map(x => x.shape),
                outputShapes: this.outputs[0].shape
            });
        }
        else {
            const outputTensor = layer.apply(this.outputs[0]);
            if (Array.isArray(outputTensor)) {
                throw new TypeError('All layers in a Sequential model ' +
                    'should have a single output tensor. ' +
                    'For multi-output layers, ' +
                    'use the functional API.');
            }
            this.checkShape(layer);
            this.outputs = [outputTensor];
            // update self.inbound_nodes
            this.inboundNodes[0].outputTensors = this.outputs;
            this.inboundNodes[0].outputShapes = [this.outputs[0].shape];
        }
        this.layers.push(layer);
        this.built = false;
    }
    /**
     * Removes the last layer in the model.
     *
     * @exception TypeError if there are no layers in the model.
     */
    pop() {
        if (this.layers.length === 0) {
            throw new TypeError('There are no layers in the model.');
        }
        this.layers.pop();
        if (this.layers.length === 0) {
            this.outputs = [];
            this.inboundNodes = [];
            this.outboundNodes = [];
        }
        else {
            const lastLayerIndex = this.layers.length - 1;
            this.layers[lastLayerIndex].outboundNodes = [];
            this.outputs = [this.layers[lastLayerIndex].output];
            // update self.inbound_nodes
            this.inboundNodes[0].outputTensors = this.outputs;
            this.inboundNodes[0].outputShapes = [this.outputs[0].shape];
        }
    }
    call(inputs, kwargs) {
        if (this.model == null) {
            this.build();
        }
        return this.model.call(inputs, kwargs);
    }
    build(inputShape) {
        // Call `getExactlyOneShape` without using its return value,
        // to verify that exactly one input shape is provided.
        getExactlyOneShape(inputShape);
        if (this.inputs.length === 0 || this.outputs.length === 0) {
            throw new TypeError('Sequential model cannot be built: model is empty.' +
                ' Add some layers first.');
        }
        // actually create the model
        this.model = new LayersModel({
            inputs: this.inputs,
            outputs: this.outputs[0],
            name: this.name + '_model'
        });
        this.model.trainable = this.trainable;
        // mirror model attributes
        this.supportsMasking = this.model.supportsMasking;
        // TODO(michaelterry): Add caches
        this.inputLayers = this.model.inputLayers;
        this.inputLayersNodeIndices = this.model.inputLayersNodeIndices;
        this.inputLayersTensorIndices = this.model.inputLayersTensorIndices;
        this.outputLayers = this.model.outputLayers;
        this.outputLayersNodeIndices = this.model.outputLayersNodeIndices;
        this.outputLayersTensorIndices = this.model.outputLayersTensorIndices;
        this.nodesByDepth = this.model.nodesByDepth;
        this.containerNodes = this.model.containerNodes;
        this.outputNames = this.model.outputNames;
        this.inputNames = this.model.inputNames;
        // TODO(michaelterry): Add feedInputNames, feedInputs, if needed.
        // TODO(michaelterry): Add callbackModel if needed.
        this.built = true;
    }
    countParams() {
        if (!this.built) {
            this.build();
        }
        return super.countParams();
    }
    /**
     * Print a text summary of the Sequential model's layers.
     *
     * The summary includes
     * - Name and type of all layers that comprise the model.
     * - Output shape(s) of the layers
     * - Number of weight parameters of each layer
     * - The total number of trainable and non-trainable parameters of the
     * model.
     *
     * ```js
     * const model = tf.sequential();
     * model.add(
     *     tf.layers.dense({units: 100, inputShape: [10], activation: 'relu'}));
     * model.add(tf.layers.dense({units: 1, activation: 'sigmoid'}));
     *
     * model.summary();
     * ```
     *
     * @param lineLength Custom line length, in number of characters.
     * @param positions Custom widths of each of the columns, as either
     *   fractions of `lineLength` (e.g., `[0.5, 0.75, 1]`) or absolute number
     *   of characters (e.g., `[30, 50, 65]`). Each number corresponds to
     *   right-most (i.e., ending) position of a column.
     * @param printFn Custom print function. Can be used to replace the default
     *   `console.log`. For example, you can use `x => {}` to mute the printed
     *   messages in the console.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    summary(lineLength, positions, printFn = console.log) {
        if (!this.built) {
            this.build();
        }
        super.summary(lineLength, positions, printFn);
    }
    /**
     * Sets the weights of the model.
     *
     * @param weights Should be a list of Tensors with shapes and types matching
     *   the output of `model.getWeights()`.
     */
    setWeights(weights) {
        if (this.model == null) {
            this.build();
        }
        this.model.setWeights(weights);
    }
    /**
     * Returns the loss value & metrics values for the model in test mode.
     *
     * Loss and metrics are specified during `compile()`, which needs to happen
     * before calls to `evaluate()`.
     *
     * Computation is done in batches.
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const result = model.evaluate(tf.ones([8, 10]), tf.ones([8, 1]), {
     *   batchSize: 4,
     * });
     * result.print();
     * ```
     *
     * @param x `tf.Tensor` of test data, or an `Array` of `tf.Tensor`s if the
     * model has multiple inputs.
     * @param y `tf.Tensor` of target data, or an `Array` of `tf.Tensor`s if the
     * model has multiple outputs.
     * @param args A `ModelEvaluateConfig`, containing optional fields.
     *
     * @return `Scalar` test loss (if the model has a single output and no
     *   metrics) or `Array` of `Scalar`s (if the model has multiple outputs
     *   and/or metrics). The attribute `model.metricsNames`
     *   will give you the display labels for the scalar outputs.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    evaluate(x, y, args = {}) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before being used.');
        }
        return this.model.evaluate(x, y, args);
    }
    // TODO(cais): Add code snippet below once real dataset objects are
    //   available.
    /**
     * Evaluate model using a dataset object.
     *
     * Note: Unlike `evaluate()`, this method is asynchronous (`async`);
     *
     * @param dataset A dataset object. Its `iterator()` method is expected
     *   to generate a dataset iterator object, the `next()` method of which
     *   is expected to produce data batches for evaluation. The return value
     *   of the `next()` call ought to contain a boolean `done` field and a
     *   `value` field. The `value` field is expected to be an array of two
     *   `tf.Tensor`s or an array of two nested `tf.Tensor` structures. The former
     *   case is for models with exactly one input and one output (e.g..
     *   a sequential model). The latter case is for models with multiple
     *   inputs and/or multiple outputs. Of the two items in the array, the
     *   first is the input feature(s) and the second is the output target(s).
     * @param args A configuration object for the dataset-based evaluation.
     * @returns Loss and metric values as an Array of `Scalar` objects.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async evaluateDataset(dataset, args) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before being used.');
        }
        return this.model.evaluateDataset(dataset, args);
    }
    /**
     * Generates output predictions for the input samples.
     *
     * Computation is done in batches.
     *
     * Note: the "step" mode of predict() is currently not supported.
     *   This is because the TensorFlow.js core backend is imperative only.
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.predict(tf.ones([2, 10])).print();
     * ```
     *
     * @param x The input data, as a Tensor, or an `Array` of `tf.Tensor`s if
     *   the model has multiple inputs.
     * @param conifg A `ModelPredictConfig` object containing optional fields.
     *
     * @return `tf.Tensor`(s) of predictions.
     *
     * @exception ValueError In case of mismatch between the provided input data
     *   and the model's expectations, or in case a stateful model receives a
     *   number of samples that is not a multiple of the batch size.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    predict(x, args = {}) {
        if (this.model == null) {
            this.build();
        }
        return this.model.predict(x, args);
    }
    /**
     * Returns predictions for a single batch of samples.
     *
     * @param x: Input samples, as a Tensor, or list of Tensors (if the model
     *   has multiple inputs).
     * @return Tensor(s) of predictions
     */
    predictOnBatch(x) {
        if (this.model == null) {
            this.build();
        }
        return this.model.predictOnBatch(x);
    }
    /**
     * See `LayersModel.compile`.
     *
     * @param args
     */
    compile(args) {
        this.build();
        this.model.compile(args);
        this.optimizer_ = this.model.optimizer;
        // tslint:disable-next-line:no-any
        this.isOptimizerOwned = this.model.isOptimizerOwned;
        this.loss = this.model.loss;
        this.metrics = this.model.metrics;
        // TODO(cais): Add this.lossWeights, this.sampleWeightMode,
        //   this.weightedMetrics, this.targets.
        this.metricsTensors = this.model.metricsTensors;
        this.metricsNames = this.model.metricsNames;
        // TODO(cais): Add sampleWeights.
    }
    get optimizer() {
        return this.model == null ? undefined : this.model.optimizer;
    }
    set optimizer(optimizer) {
        this.model.optimizer = optimizer;
    }
    /**
     * Trains the model for a fixed number of epochs (iterations on a dataset).
     *
     * ```js
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [10]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const history = await model.fit(tf.ones([8, 10]), tf.ones([8, 1]), {
     *   batchSize: 4,
     *   epochs: 3
     * });
     * console.log(history.history.loss[0]);
     * ```
     *
     * @param x `tf.Tensor` of training data, or an array of `tf.Tensor`s if the
     * model has multiple inputs. If all inputs in the model are named, you can
     * also pass a dictionary mapping input names to `tf.Tensor`s.
     * @param y `tf.Tensor` of target (label) data, or an array of `tf.Tensor`s if
     * the model has multiple outputs. If all outputs in the model are named, you
     *  can also pass a dictionary mapping output names to `tf.Tensor`s.
     * @param args  A `ModelFitConfig`, containing optional fields.
     *
     * @return A `History` instance. Its `history` attribute contains all
     *   information collected during training.
     *
     * @exception ValueError In case of mismatch between the provided input data
     *   and what the model expects.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async fit(x, y, args = {}) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before ' +
                'being used.');
        }
        return this.model.fit(x, y, args);
    }
    /**
     * Trains the model using a dataset object.
     *
     * ```js
     * const xArray = [
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     *   [1, 1, 1, 1, 1, 1, 1, 1, 1],
     * ];
     * const yArray = [1, 1, 1, 1];
     * // Create a dataset from the JavaScript array.
     * const xDataset = tf.data.array(xArray);
     * const yDataset = tf.data.array(yArray);
     * // Zip combines the `x` and `y` Datasets into a single Dataset, the
     * // iterator of which will return an object containing of two tensors,
     * // corresponding to `x` and `y`.  The call to `batch(4)` will bundle
     * // four such samples into a single object, with the same keys now pointing
     * // to tensors that hold 4 examples, organized along the batch dimension.
     * // The call to `shuffle(4)` causes each iteration through the dataset to
     * // happen in a different order.  The size of the shuffle window is 4.
     * const xyDataset = tf.data.zip({xs: xDataset, ys: yDataset})
     *     .batch(4)
     *     .shuffle(4);
     * const model = tf.sequential({
     *   layers: [tf.layers.dense({units: 1, inputShape: [9]})]
     * });
     * model.compile({optimizer: 'sgd', loss: 'meanSquaredError'});
     * const history = await model.fitDataset(xyDataset, {
     *   epochs: 4,
     *   callbacks: {onEpochEnd: (epoch, logs) => console.log(logs.loss)}
     * });
     * ```
     *
     * @param dataset A dataset object. Its `iterator()` method is expected to
     *   generate a dataset iterator object, the `next()` method of which is
     *   expected to produce data batches for evaluation. The return value of the
     *   `next()` call ought to contain a boolean `done` field and a `value`
     *   field.
     *
     *   The `value` field is expected to be an object of with fields
     *   `xs` and `ys`, which point to the feature tensor and the target tensor,
     *   respectively. This case is for models with exactly one input and one
     *   output (e.g.. a sequential model). For example:
     *   ```js
     *   {value: {xs: xsTensor, ys: ysTensor}, done: false}
     *   ```
     *
     *   If the model has multiple inputs, the `xs` field of `value` should
     *   be an object mapping input names to their respective feature tensors.
     *   For example:
     *   ```js
     *   {
     *     value: {
     *       xs: {
     *         input_1: xsTensor1,
     *         input_2: xsTensor2
     *       },
     *       ys: ysTensor
     *     },
     *     done: false
     *   }
     *   ```
     *   If the model has multiple outputs, the `ys` field of `value` should
     *   be an object mapping output names to their respective target tensors.
     *   For example:
     *   ```js
     *   {
     *     value: {
     *       xs: xsTensor,
     *       ys: {
     *         output_1: ysTensor1,
     *         output_2: ysTensor2
     *       },
     *     },
     *     done: false
     *   }
     *   ```
     * @param args A `ModelFitDatasetArgs`, containing optional fields.
     *
     * @return A `History` instance. Its `history` attribute contains all
     *   information collected during training.
     *
     * @doc {heading: 'Models', subheading: 'Classes', ignoreCI: true}
     */
    async fitDataset(dataset, args) {
        if (!this.built) {
            throw new RuntimeError('The model needs to be compiled before ' +
                'being used.');
        }
        return this.model.fitDataset(dataset, args);
    }
    /**
     * Runs a single gradient update on a single batch of data.
     *
     * This method differs from `fit()` and `fitDataset()` in the following
     * regards:
     *   - It operates on exactly one batch of data.
     *   - It returns only the loss and matric values, instead of
     *     returning the batch-by-batch loss and metric values.
     *   - It doesn't support fine-grained options such as verbosity and
     *     callbacks.
     *
     * @param x Input data. It could be one of the following:
     *   - A `tf.Tensor`, or an Array of `tf.Tensor`s (in case the model has
     *     multiple inputs).
     *   - An Object mapping input names to corresponding `tf.Tensor` (if the
     *     model has named inputs).
     * @param y Target darta. It could be either a `tf.Tensor` a multiple
     *   `tf.Tensor`s. It should be consistent with `x`.
     * @returns Training loss or losses (in case the model has
     *   multiple outputs), along with metrics (if any), as numbers.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async trainOnBatch(x, y) {
        return this.model.trainOnBatch(x, y);
    }
    /* See parent class for JsDoc */
    /** @nocollapse */
    static fromConfig(cls, config, customObjects = {}, fastWeightInit = false) {
        let configArray;
        let extraModelConfig = {};
        if (config instanceof Array) {
            if (!(config[0].className != null) ||
                config[0]['className'] === 'Merge') {
                throw new ValueError('Legacy serialization format not supported yet.');
            }
            configArray = config;
        }
        else {
            util.assert(config['layers'] != null, () => `When the config data for a Sequential model is not an Array, ` +
                `it must be an Object that contains the 'layers' field.`);
            configArray = config['layers'];
            delete config['layers'];
            extraModelConfig = config;
        }
        const model = new cls(extraModelConfig);
        if (!(model instanceof Sequential)) {
            throw new NotImplementedError(`Sequential.fromConfig called on non-Sequential input: ${model}`);
        }
        for (const conf of configArray) {
            const customObjects = undefined;
            const layer = deserialize(conf, customObjects, fastWeightInit);
            if (fastWeightInit) {
                layer.setFastWeightInitDuringBuild(true);
            }
            model.add(layer);
        }
        return model;
    }
    /**
     * Setter used for force stopping of LayersModel.fit() (i.e., training).
     *
     * Example:
     *
     * ```js
     * const model = tf.sequential();
     * model.add(tf.layers.dense({units: 1, inputShape: [10]}));
     * model.compile({loss: 'meanSquaredError', optimizer: 'sgd'});
     * const xs = tf.ones([8, 10]);
     * const ys = tf.zeros([8, 1]);
     *
     * const history = await model.fit(xs, ys, {
     *   epochs: 10,
     *   callbacks: {
     *     onEpochEnd: async (epoch, logs) => {
     *       if (epoch === 2) {
     *         model.stopTraining = true;
     *       }
     *     }
     *   }
     * });
     *
     * // There should be only 3 values in the loss array, instead of 10 values,
     * // due to the stopping after 3 epochs.
     * console.log(history.history.loss);
     * ```
     */
    set stopTraining(stop) {
        // TODO(cais): When refactoring to remove the composition pattern happens,
        // remove this method overriding.
        if (this.model == null) {
            throw new ValueError('Cannot set the stopTraining property of a sequential model before ' +
                'it is compiled.');
        }
        this.model.stopTraining = stop;
    }
    get stopTraining() {
        if (this.model == null) {
            throw new ValueError('Cannot get the stopTraining property of a sequential model before ' +
                'it is compiled.');
        }
        return this.model.stopTraining;
    }
    // TODO(cais): Override get trainableWeights() here
    // tslint:disable-next-line:no-any
    getConfig() {
        // NOTE(cais): We override the return type of getConfig() to `any` here,
        //   because the `Sequential` class is a special case among `Container`
        //   subtypes in that its getConfig() method returns an Array (not a
        //   dict).
        const layers = [];
        for (const layer of this.layers) {
            const dict = {};
            dict['className'] = layer.getClassName();
            dict['config'] = layer.getConfig();
            layers.push(dict);
        }
        return { name: this.name, layers };
    }
}
/** @nocollapse */
Sequential.className = 'Sequential';
serialization.registerClass(Sequential);
//# sourceMappingURL=data:application/json;base64,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