/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * TensorFlow.js Layers: Depthwise Convolutional Layers
 */
import * as tfc from '@tensorflow/tfjs-core';
import { serialization, tidy } from '@tensorflow/tfjs-core';
import { imageDataFormat } from '../backend/common';
import * as K from '../backend/tfjs_backend';
import { checkDataFormat } from '../common';
import { getConstraint, serializeConstraint } from '../constraints';
import { ValueError } from '../errors';
import { getInitializer, serializeInitializer } from '../initializers';
import { getRegularizer, serializeRegularizer } from '../regularizers';
import { convOutputLength } from '../utils/conv_utils';
import { getExactlyOneShape, getExactlyOneTensor } from '../utils/types_utils';
import { BaseConv, preprocessConv2DInput } from './convolutional';
/**
 * 2D convolution with separable filters.
 * @param x Input tensor.
 * @param depthwiseKernel Convolution kernel for depthwise convolution.
 * @param strides Strides (Array of two integers).
 * @param padding Padding model.
 * @param dataFormat Data format.
 * @param dilationRate Array of two integers, dilation rates for the separable
 *   convolution.
 * @returns Output tensor.
 * @throws ValueError If depthwiseKernel is not a 4D array.
 */
export function depthwiseConv2d(x, depthwiseKernel, strides = [1, 1], padding = 'valid', dataFormat, dilationRate) {
    return tidy(() => {
        if (dataFormat == null) {
            dataFormat = imageDataFormat();
        }
        checkDataFormat(dataFormat);
        let y = preprocessConv2DInput(x, dataFormat);
        if (x.rank !== 4) {
            throw new ValueError(`Input for depthwiseConv2d is required to be 4-D, but is instead ` +
                `${x.rank}-D`);
        }
        if (depthwiseKernel.rank !== 4) {
            throw new ValueError(`depthwiseKernel is required to be 4-D, but is instead ` +
                `${depthwiseKernel.rank}-D`);
        }
        y = tfc.depthwiseConv2d(y, depthwiseKernel, strides, padding === 'same' ? 'same' : 'valid', 'NHWC', dilationRate);
        if (dataFormat === 'channelsFirst') {
            y = tfc.transpose(y, [0, 3, 1, 2]);
        }
        return y;
    });
}
export class DepthwiseConv2D extends BaseConv {
    constructor(args) {
        super(2, args);
        this.depthwiseKernel = null;
        this.depthMultiplier =
            args.depthMultiplier == null ? 1 : args.depthMultiplier;
        this.depthwiseInitializer = getInitializer(args.depthwiseInitializer || this.DEFAULT_KERNEL_INITIALIZER);
        this.depthwiseConstraint = getConstraint(args.depthwiseConstraint);
        this.depthwiseRegularizer = getRegularizer(args.depthwiseRegularizer);
    }
    build(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        if (inputShape.length < 4) {
            throw new ValueError(`Inputs to DepthwiseConv2D should have rank 4. ` +
                `Received input shape: ${JSON.stringify(inputShape)}.`);
        }
        const channelAxis = this.dataFormat === 'channelsFirst' ? 1 : 3;
        if (inputShape[channelAxis] == null || inputShape[channelAxis] < 0) {
            throw new ValueError('The channel dimension of the inputs to DepthwiseConv2D should ' +
                `be defined, but is not (${inputShape[channelAxis]}).`);
        }
        const inputDim = inputShape[channelAxis];
        const depthwiseKernelShape = [
            this.kernelSize[0], this.kernelSize[1], inputDim, this.depthMultiplier
        ];
        this.depthwiseKernel = this.addWeight('depthwise_kernel', depthwiseKernelShape, null, this.depthwiseInitializer, this.depthwiseRegularizer, true, this.depthwiseConstraint);
        if (this.useBias) {
            this.bias = this.addWeight('bias', [inputDim * this.depthMultiplier], null, this.biasInitializer, this.biasRegularizer, true, this.biasConstraint);
        }
        else {
            this.bias = null;
        }
        this.built = true;
    }
    call(inputs, kwargs) {
        return tidy(() => {
            inputs = getExactlyOneTensor(inputs);
            let outputs = depthwiseConv2d(inputs, this.depthwiseKernel.read(), this.strides, this.padding, this.dataFormat, null);
            // TODO(cais): Add support for dilation.
            if (this.useBias) {
                outputs = K.biasAdd(outputs, this.bias.read(), this.dataFormat);
            }
            if (this.activation != null) {
                outputs = this.activation.apply(outputs);
            }
            return outputs;
        });
    }
    computeOutputShape(inputShape) {
        inputShape = getExactlyOneShape(inputShape);
        const rows = this.dataFormat === 'channelsFirst' ? inputShape[2] : inputShape[1];
        const cols = this.dataFormat === 'channelsFirst' ? inputShape[3] : inputShape[2];
        const outFilters = this.dataFormat === 'channelsFirst' ?
            inputShape[1] * this.depthMultiplier :
            inputShape[3] * this.depthMultiplier;
        const outRows = convOutputLength(rows, this.kernelSize[0], this.padding, this.strides[0]);
        const outCols = convOutputLength(cols, this.kernelSize[1], this.padding, this.strides[1]);
        if (this.dataFormat === 'channelsFirst') {
            return [inputShape[0], outFilters, outRows, outCols];
        }
        else {
            // In this case, assume 'channelsLast'.
            return [inputShape[0], outRows, outCols, outFilters];
        }
    }
    getConfig() {
        const config = super.getConfig();
        config['depthMultiplier'] = this.depthMultiplier;
        config['depthwiseInitializer'] =
            serializeInitializer(this.depthwiseInitializer);
        config['depthwiseRegularizer'] =
            serializeRegularizer(this.depthwiseRegularizer);
        config['depthwiseConstraint'] =
            serializeConstraint(this.depthwiseRegularizer);
        return config;
    }
}
/** @nocollapse */
DepthwiseConv2D.className = 'DepthwiseConv2D';
serialization.registerClass(DepthwiseConv2D);
//# sourceMappingURL=data:application/json;base64,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