/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * Interfaces and methods for training models using TensorFlow.js datasets.
 */
import * as tfc from '@tensorflow/tfjs-core';
import { scalar } from '@tensorflow/tfjs-core';
import { configureCallbacks, standardizeCallbacks } from '../base_callbacks';
import { NotImplementedError, ValueError } from '../errors';
import { disposeTensorsInLogs } from '../logs';
import { singletonOrArray, toList } from '../utils/generic_utils';
import { standardizeClassWeights, standardizeWeights } from './training_utils';
// Default batch size used during tensor-based validation.
const DEFAULT_VALIDATION_BATCH_SIZE = 32;
/**
 * Standardize the output of a dataset iterator for use by
 * LayersModel.fitDataset().
 *
 * @param model: A `tf.LayersModel` object.
 * @param iteratorOut The output of a dataset iterator. It is required to be
 *   an object of the form `{xs: TensorOrArrayOrMap, ys:
 * TensorOrArrayOrMap}`, where `TensorOrArrayOrMap` is a single `tf.Tensor`,
 * a `tf.Tensor[]`, or a flat map from string names to `tf.Tensor`s.
 * @returns A flat array of `tf.Tensor` objects: the input `tf.Tensor`s
 *   followed by the target `tf.Tensor`s.  When `tf.Tensor`s are provided
 *   as a map, the order in the resulting array is taken from the `inputNames`
 *   and `outputNames` of the model.
 */
function standardizeDataIteratorOutput(
// Type `model` as `any` here to avoid circular dependency w/
// training.ts.
// tslint:disable-next-line:no-any
model, iteratorOut) {
    let xs;
    let ys;
    const iteratorOutObj = iteratorOut;
    xs = iteratorOutObj['xs'];
    ys = iteratorOutObj['ys'];
    tfc.util.assert(xs != null && ys != null, () => 'A Dataset iterator for fitDataset() is expected to generate ' +
        'objects of the form `{xs: xVal, ys: yVal}`, where the two ' +
        'values may be `tf.Tensor`, an array of Tensors, or a map of ' +
        'string to Tensor.  The provided Dataset instead generates ' +
        `${iteratorOut}`);
    const flattenedXs = flattenTensorOrArrayOrMap('input', model.inputNames, xs);
    const flattenedYs = flattenTensorOrArrayOrMap('output', model.outputNames, ys);
    const batchSize = flattenedXs[0].shape[0];
    tfc.util.assert(flattenedXs.length === model.inputs.length, () => `LayersModel has ${model.inputs.length} inputs, but the dataset ` +
        `provides ${flattenedXs.length} inputs.  (Expected input keys: ` +
        `${JSON.stringify(model.inputNames)})`);
    tfc.util.assert(flattenedYs.length === model.outputs.length, () => `LayersModel has ${model.outputs.length} outputs, but the dataset ` +
        `provides ${flattenedYs.length} outputs.  (Expected output keys: ` +
        `${JSON.stringify(model.outputNames)})`);
    for (let xIndex = 0; xIndex < flattenedXs.length; xIndex++) {
        tfc.util.assert(flattenedXs[xIndex].shape[0] === batchSize, () => `Batch size mismatch: input ` +
            `${model.inputNames[xIndex]} has ${flattenedXs[xIndex].shape[0]}; ` +
            `expected  ${batchSize} based on input ${model.inputNames[0]}.`);
    }
    for (let yIndex = 0; yIndex < flattenedYs.length; yIndex++) {
        tfc.util.assert(flattenedYs[yIndex].shape[0] === batchSize, () => `Batch size mismatch: output ` +
            `${model.outputNames[yIndex]} has ${flattenedYs[yIndex].shape[0]}; ` +
            `expected  ${batchSize} based on input ${model.inputNames[0]}.`);
    }
    return { xs: flattenedXs, ys: flattenedYs };
}
function flattenTensorOrArrayOrMap(inputOrOutput, names, values) {
    if (values instanceof tfc.Tensor) {
        return [values];
    }
    else if (Array.isArray(values)) {
        tfc.util.assert(values.length === names.length, () => `Received an array of ${values.length} Tensors, but expected ${names.length} to match the ${inputOrOutput} keys ${names}.`);
        return values;
    }
    else {
        const result = [];
        // Check that all the required keys are available.
        for (const name of names) {
            if (values[name] == null) {
                throw new ValueError(`The feature data generated by the dataset lacks the required ` +
                    `${inputOrOutput} key '${name}'.`);
            }
            result.push(values[name]);
        }
        return result;
    }
}
function standardizeTensorValidationData(data) {
    if (data.length === 3) {
        throw new NotImplementedError('Validation with sample weights is not implemented yet.');
    }
    return { xs: data[0], ys: data[1] };
}
export async function fitDataset(
// Type `model` as `any` here to avoid circular dependency w/
// training.ts.
// tslint:disable-next-line:no-any
model, dataset, args) {
    const hasBatchesPerEpoch = args.batchesPerEpoch != null;
    tfc.util.assert(model.optimizer != null, () => 'You must compile a model before training/testing. Use ' +
        'LayersModel.compile(modelCompileConfig).');
    tfc.util.assert(args != null, () => `For fitDataset(), the 2nd argument (config) is required, ` +
        `but it is not provided in this call.`);
    tfc.util.assert(args.epochs != null && args.epochs > 0 && Number.isInteger(args.epochs), () => `For fitDataset(), config.epochs is expected to be a positive ` +
        `integer, but got ${args.epochs}`);
    tfc.util.assert(!hasBatchesPerEpoch ||
        (args.batchesPerEpoch > 0 && Number.isInteger(args.batchesPerEpoch)), () => `For fitDataset(), config.batchesPerEpoch is expected to be a ` +
        `positive integer if specified, but got ${args.batchesPerEpoch}`);
    tfc.util.assert(
    // tslint:disable-next-line:no-any
    args['validationSplit'] == null, () => '`validationSplit` is not supported by `fitDataset()`. ' +
        'Use validationData instead.');
    if (model.isTraining) {
        throw new Error('Cannot start training because another fit() call is ongoing.');
    }
    model.isTraining = true;
    try {
        const doValidation = args.validationData != null;
        let valXs;
        let valYs;
        if (doValidation) {
            if (isDatasetObject(args.validationData)) {
                tfc.util.assert(args.validationBatches == null ||
                    (args.validationBatches > 0 &&
                        Number.isInteger(args.validationBatches)), () => `For fitDataset() with dataset-based validation, ` +
                    `config.validationBatches is expected not to be provided, ` +
                    `or to be a positive integer, ` +
                    `but got ${args.validationBatches}`);
            }
            else {
                const validationData = standardizeTensorValidationData(args.validationData);
                valXs = validationData.xs;
                valYs = validationData.ys;
            }
        }
        const trainFunction = model.makeTrainFunction();
        const outLabels = model.getDedupedMetricsNames();
        let callbackMetrics;
        if (doValidation) {
            callbackMetrics =
                outLabels.slice().concat(outLabels.map(n => 'val_' + n));
        }
        else {
            callbackMetrics = outLabels.slice();
        }
        const callbacks = standardizeCallbacks(args.callbacks, args.yieldEvery);
        const verbose = args.verbose == null ? 1 : args.verbose;
        const { callbackList, history } = configureCallbacks(callbacks, verbose, args.epochs, null, null, getStepsPerEpoch(dataset, args), null, // Batch size determined by the dataset itself.
        doValidation, callbackMetrics);
        callbackList.setModel(model);
        model.history = history;
        await callbackList.onTrainBegin();
        model.stopTraining_ = false;
        let epoch = args.initialEpoch == null ? 0 : args.initialEpoch;
        let dataIterator = await dataset.iterator();
        while (epoch < args.epochs) {
            const epochLogs = {};
            await callbackList.onEpochBegin(epoch);
            let stepsDone = 0;
            let batchIndex = 0;
            if (!hasBatchesPerEpoch) {
                dataIterator = await dataset.iterator();
            }
            while (hasBatchesPerEpoch ? stepsDone < args.batchesPerEpoch : true) {
                const iteratorOut = await dataIterator.next();
                // If `batchesPerEpoch` is specified, the dataset should not be
                // exhausted until all epoches are done.
                if (hasBatchesPerEpoch && iteratorOut.done) {
                    console.warn('You provided `batchesPerEpoch` as ' +
                        `${args.batchesPerEpoch}, ` +
                        'but your dataset iterator ran out of data after ' +
                        `${stepsDone} batches; ` +
                        'interrupting training. Make sure that your ' +
                        'dataset can generate at least `batchesPerEpoch * epochs` ' +
                        'batches (in this case, ' +
                        `${args.batchesPerEpoch * args.epochs} batches). ` +
                        'You may need to use the repeat() function when building ' +
                        'your dataset.');
                    break;
                }
                if (iteratorOut.value != null) {
                    const { xs, ys } = standardizeDataIteratorOutput(model, iteratorOut.value);
                    const batchLogs = {};
                    batchLogs['batch'] = batchIndex;
                    batchLogs['size'] = xs[0].shape[0];
                    await callbackList.onBatchBegin(batchIndex, batchLogs);
                    const sampleWeights = [];
                    if (args.classWeight != null) {
                        const standardClassWeights = standardizeClassWeights(args.classWeight, model.outputNames);
                        for (let i = 0; i < standardClassWeights.length; ++i) {
                            sampleWeights.push(await standardizeWeights(ys[i], null, standardClassWeights[i]));
                        }
                    }
                    // Train on batch.
                    const ins = xs.concat(ys).concat(sampleWeights);
                    const outs = trainFunction(ins);
                    tfc.dispose(ins);
                    for (let i = 0; i < outLabels.length; ++i) {
                        const label = outLabels[i];
                        const out = outs[i];
                        batchLogs[label] = out;
                        tfc.keep(out);
                    }
                    await callbackList.onBatchEnd(batchIndex, batchLogs);
                    disposeTensorsInLogs(batchLogs);
                    batchIndex++;
                    stepsDone++;
                }
                if (hasBatchesPerEpoch ? stepsDone >= args.batchesPerEpoch :
                    iteratorOut.done) {
                    // Epoch finished. Perform validation.
                    if (doValidation) {
                        let valOuts;
                        if (isDatasetObject(args.validationData)) {
                            valOuts = toList(await model.evaluateDataset(args.validationData, { batches: args.validationBatches }));
                        }
                        else {
                            valOuts = toList(model.evaluate(valXs, valYs, {
                                batchSize: args.validationBatchSize == null ?
                                    DEFAULT_VALIDATION_BATCH_SIZE :
                                    args.validationBatchSize,
                                verbose: 0
                            }));
                        }
                        for (let i = 0; i < model.metricsNames.length; ++i) {
                            epochLogs[`val_${model.metricsNames[i]}`] = valOuts[i];
                        }
                    }
                    // Call `break` to exit one epoch lopp after validation is done. If
                    // config.batchesPerEpoch is specified, an epoch while loop will
                    // stop when `stepsDone >= config.batchesPerEpoch`. When
                    // config.batchesPerEpoch is not provided, the following `break` is
                    // required to exit the while lopp after dataset is exhausted.
                    break;
                }
                if (model.stopTraining_) {
                    break;
                }
            }
            await callbackList.onEpochEnd(epoch, epochLogs);
            epoch++;
            if (model.stopTraining_) {
                break;
            }
        }
        await callbackList.onTrainEnd();
        await model.history.syncData();
        return model.history;
    }
    finally {
        model.isTraining = false;
    }
}
/** Helper function that determines number of steps (batches) per epoch. */
function getStepsPerEpoch(dataset, args) {
    // Attempt to determine # of batches in an epoch.
    let stepsPerEpoch = null;
    if (args.batchesPerEpoch != null) {
        stepsPerEpoch = args.batchesPerEpoch;
    }
    else if (Number.isFinite(dataset.size)) {
        stepsPerEpoch = dataset.size;
    }
    return stepsPerEpoch;
}
// Check if provided object is a Dataset object by checking its .iterator
// element.
function isDatasetObject(dataset) {
    return (typeof dataset.iterator === 'function');
}
// Check if provided object is a LazyIterator object by checking it's .next
// element.
function isLazyIteratorObject(iterator) {
    return (typeof iterator.next === 'function');
}
export async function evaluateDataset(
// Type `model` as `any` here to avoid circular dependency w/
// training.ts.
// tslint:disable-next-line:no-any
model, dataset, args) {
    args = args || {};
    const hasBatches = args.batches != null;
    const f = model.testFunction;
    let outs = [];
    if (args.verbose > 0) {
        throw new NotImplementedError('Verbose mode is not implemented yet.');
    }
    tfc.util.assert(!hasBatches || (args.batches > 0 && Number.isInteger(args.batches)), () => 'Test loop expects `batches` to be a positive integer, but ' +
        `received ${JSON.stringify(args.batches)}`);
    const dataIterator = isLazyIteratorObject(dataset) ?
        dataset :
        await dataset.iterator();
    // Keeps track of number of examples used in this evaluation.
    let numExamples = 0;
    let batch = 0;
    while (hasBatches ? batch < args.batches : true) {
        const iteratorOut = await dataIterator.next();
        outs = tfc.tidy(() => {
            if (iteratorOut.value) {
                // TODO(cais): Once real dataset is available, use
                //   `map(x => standardizeDataIteratorOutput(model, x).map(f)`.
                const { xs, ys } = standardizeDataIteratorOutput(model, iteratorOut.value);
                const xsAndYs = xs.concat(ys);
                const batchOuts = tfc.tidy(() => f(xsAndYs));
                tfc.dispose(xsAndYs);
                if (batch === 0) {
                    for (let i = 0; i < batchOuts.length; ++i) {
                        outs.push(scalar(0));
                    }
                }
                const batchSize = xsAndYs[0].shape[0];
                for (let i = 0; i < batchOuts.length; ++i) {
                    const batchOut = batchOuts[i];
                    const oldScalar = outs[i];
                    outs[i] =
                        tfc.tidy(() => tfc.add(outs[i], tfc.mul(batchSize, batchOut)));
                    if (batch > 0) {
                        tfc.dispose(oldScalar);
                    }
                }
                tfc.dispose(batchOuts);
                numExamples += batchSize;
                ++batch;
            }
            return outs;
        });
        if (iteratorOut.done) {
            if (hasBatches) {
                console.warn('Your dataset iterator ran out of data during evaluateDataset(). ' +
                    'Interrupting evalution. Make sure that your ' +
                    'dataset can generate at least `batches` ' +
                    `batches (in this case, ${args.batches} batches). ` +
                    'You may need to use the repeat() function when building ' +
                    'your dataset.');
            }
            break;
        }
    }
    for (let i = 0; i < outs.length; ++i) {
        const oldScalar = outs[i];
        outs[i] = tfc.div(outs[i], numExamples);
        tfc.dispose(oldScalar);
    }
    return singletonOrArray(outs);
}
//# sourceMappingURL=data:application/json;base64,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