/**
 * @license
 * Copyright 2018 Google LLC
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE file or at
 * https://opensource.org/licenses/MIT.
 * =============================================================================
 */
/**
 * Executor: Evaluates SymbolicTensor based on feeds.
 */
import { cast, dispose, memory, util } from '@tensorflow/tfjs-core';
import { ValueError } from '../errors';
import { LruCache } from '../utils/executor_utils';
import { toList } from '../utils/generic_utils';
import { InputLayer } from './input_layer';
import { SymbolicTensor } from './topology';
/**
 * Helper function to check the dtype and shape compatibility of a feed value.
 */
function assertFeedCompatibility(key, val) {
    // Check dtype compatibility.
    if (key.dtype == null || key.dtype === val.dtype) {
        //  a.  If types match, return val tensor as is.
        return val;
    }
    try {
        //  b. Attempt to convert to expected type.
        return cast(val, key.dtype);
    }
    catch (err) {
        //  c. If conversion fails, return helpful error.
        throw new ValueError(`The dtype of the feed (${val.dtype}) can not be cast to the dtype ` +
            `of the key '${key.name}' (${key.dtype}).`);
    }
}
/**
 * FeedDict: A mapping from unique SymbolicTensors to feed values for them.
 * A feed value is a concrete value represented as an `Tensor`.
 */
export class FeedDict {
    /**
     * Constructor, optionally does copy-construction.
     * @param feeds An Array of `Feed`s, or another `FeedDict`, in which case
     *   copy-construction will be performed.
     */
    constructor(feeds) {
        this.id2Value = {};
        this.id2Mask = {};
        this.name2Id = {};
        if (feeds instanceof FeedDict) {
            for (const id in feeds.id2Value) {
                this.id2Value[id] = feeds.id2Value[id];
                if (id in feeds.id2Mask) {
                    this.id2Mask[id] = feeds.id2Mask[id];
                }
            }
        }
        else {
            if (feeds == null) {
                return;
            }
            for (const feed of feeds) {
                this.add(feed.key, feed.value);
            }
        }
    }
    /**
     * Add a key-value pair to the FeedDict.
     *
     * @param key The key of the feed.
     * @param value The value of the tensor feed.
     * @param mask The value of the mask feed (optional).
     * @returns This `FeedDict`.
     * @throws ValueError: If the key `SymbolicTensor` already exists in the
     *   `FeedDict`.
     */
    add(key, value, mask) {
        if (this.id2Value[key.id] == null) {
            this.id2Value[key.id] = assertFeedCompatibility(key, value);
            this.name2Id[key.name] = key.id;
            if (mask != null) {
                this.id2Mask[key.id] = mask;
            }
        }
        else {
            throw new ValueError(`Duplicate key: name=${key.name}, id=${key.id}`);
        }
        return this;
    }
    /**
     * Add a Feed to the FeedDict.
     * @param feed The new `Feed` to add.
     * @returns This `FeedDict`.
     */
    addFeed(feed) {
        this.add(feed.key, feed.value);
    }
    /**
     * Probe whether a key already exists in the FeedDict.
     * @param key
     */
    hasKey(key) {
        return this.id2Value[key.id] != null;
    }
    /**
     * Get all the SymbolicTensor available in this FeedDict.
     */
    names() {
        return Object.keys(this.name2Id);
    }
    /**
     * Get the feed value for given key.
     * @param key The SymbolicTensor, or its name (as a string), of which the
     *     value is sought.
     * @returns If `key` exists, the corresponding feed value.
     * @throws ValueError: If `key` does not exist in this `FeedDict`.
     */
    getValue(key) {
        if (key instanceof SymbolicTensor) {
            if (this.id2Value[key.id] == null) {
                throw new ValueError(`Nonexistent key: ${key.name}`);
            }
            else {
                return this.id2Value[key.id];
            }
        }
        else {
            const id = this.name2Id[key];
            if (id == null) {
                throw new ValueError(`Feed dict has no SymbolicTensor name: ${key}`);
            }
            return this.id2Value[id];
        }
    }
    /**
     * Get the feed mask for given key.
     * @param key The SymbolicTensor, or its name (as a string), of which the
     *     value is sought.
     * @returns If `key` exists, the corresponding feed mask.
     * @throws ValueError: If `key` does not exist in this `FeedDict`.
     */
    getMask(key) {
        if (key instanceof SymbolicTensor) {
            if (this.id2Value[key.id] == null) {
                throw new ValueError(`Nonexistent key: ${key.name}`);
            }
            else {
                return this.id2Mask[key.id];
            }
        }
        else {
            const id = this.name2Id[key];
            if (id == null) {
                throw new ValueError(`Feed dict has no SymbolicTensor name: ${key}`);
            }
            return this.id2Mask[id];
        }
    }
    /** Dispose all mask Tensors held by this object. */
    disposeMasks() {
        if (this.id2Mask != null) {
            dispose(this.id2Mask);
        }
    }
}
// Cache for topologically sorted SymbolicTensors for given execution
// targets (i.e., fetches).
export const cachedSorted = new LruCache();
// Cache for recipient count maps for given execution targets (i.e., fetches).
export const cachedRecipientCounts = new LruCache();
export function updateCacheMaxEntries(maxEntries) {
    if (cachedSorted != null) {
        cachedSorted.setMaxEntries(maxEntries);
    }
    if (cachedRecipientCounts != null) {
        cachedRecipientCounts.setMaxEntries(maxEntries);
    }
}
/**
 * Execute a SymbolicTensor by using concrete feed values.
 *
 * A `SymbolicTensor` object is a node in a computation graph of TF.js
 * Layers. The object is backed by a source layer and input
 * `SymbolicTensor`s to the source layer. This method evaluates
 * the `call()` method of the source layer, using concrete values of the
 * inputs obtained from either
 * * `feedDict`, if the input key exists in `feedDict`, or else,
 * * a recursive call to `execute()` itself.
 *
 * @param x: The `SymbolicTensor` to execute.
 * @param feedDict: The feed values, as base condition of the recursion.
 *   execution.
 * @param kwargs: Optional keyword arguments.
 * @param probe: A probe object (of interface `ExecutionProbe`) used for
 *   testing memory footprint of `execute` calls.
 * @returns Result of the execution.
 * @throws ValueError: If any `SymbolicTensor`s from `InputLayer`s
 *   encountered during the execution lacks a feed value in `feedDict`.
 */
export function execute(fetches, feedDict, kwargs, probe) {
    const training = kwargs == null ? false : kwargs['training'];
    const arrayFetches = Array.isArray(fetches);
    const fetchArray = arrayFetches ? fetches : [fetches];
    const outputNames = fetchArray.map(t => t.name);
    const finalOutputs = [];
    const feedNames = feedDict.names();
    for (const outputName of outputNames) {
        if (feedNames.indexOf(outputName) !== -1) {
            finalOutputs.push(feedDict.getValue(outputName));
        }
        else {
            finalOutputs.push(null);
        }
    }
    if (probe != null) {
        // For optional probing of memory footprint during execution.
        probe.maxNumTensors = -Infinity;
        probe.minNumTensors = Infinity;
    }
    // Check cache.
    const fetchAndFeedKey = outputNames.join(',') + '|' + feedDict.names().sort().join(',');
    let sorted = cachedSorted.get(fetchAndFeedKey);
    let recipientCounts;
    if (sorted == null) {
        // Cache doesn't contain the desired combination of fetches. Compute
        // topological sort for the combination for the first time.
        const out = getTopologicalSortAndRecipientCounts(fetchArray, feedDict);
        sorted = out.sorted;
        recipientCounts = out.recipientCounts;
        // Store results in cache for future use.
        cachedSorted.put(fetchAndFeedKey, sorted);
        cachedRecipientCounts.put(fetchAndFeedKey, recipientCounts);
    }
    recipientCounts = {};
    if (!training) {
        Object.assign(recipientCounts, cachedRecipientCounts.get(fetchAndFeedKey));
    }
    const internalFeedDict = new FeedDict(feedDict);
    // Start iterative execution on the topologically-sorted SymbolicTensors.
    for (let i = 0; i < sorted.length; ++i) {
        if (probe != null) {
            // For optional probing of memory usage during execution.
            const numTensors = memory().numTensors;
            if (numTensors > probe.maxNumTensors) {
                probe.maxNumTensors = numTensors;
            }
            if (numTensors < probe.minNumTensors) {
                probe.minNumTensors = numTensors;
            }
        }
        const symbolic = sorted[i];
        const srcLayer = symbolic.sourceLayer;
        if (srcLayer instanceof InputLayer) {
            continue;
        }
        const inputValues = [];
        const inputMasks = [];
        const tensorsToDispose = [];
        let maskExists = false;
        for (const input of symbolic.inputs) {
            const value = internalFeedDict.getValue(input);
            const mask = internalFeedDict.getMask(input);
            inputValues.push(value);
            inputMasks.push(mask);
            if (mask != null) {
                maskExists = true;
            }
            if (!training) {
                recipientCounts[input.name]--;
                if (recipientCounts[input.name] === 0 && !feedDict.hasKey(input) &&
                    outputNames.indexOf(input.name) === -1 && !value.isDisposed &&
                    input.sourceLayer.stateful !== true) {
                    tensorsToDispose.push(value);
                }
            }
        }
        if (maskExists) {
            kwargs = kwargs || {};
            kwargs['mask'] = inputMasks[0];
        }
        const outputTensors = toList(srcLayer.apply(inputValues, kwargs));
        let outputMask = null;
        if (srcLayer.supportsMasking) {
            outputMask = srcLayer.computeMask(inputValues, inputMasks);
        }
        const layerOutputs = getNodeOutputs(symbolic);
        const outputSymbolicTensors = Array.isArray(layerOutputs) ? layerOutputs : [layerOutputs];
        for (let i = 0; i < outputSymbolicTensors.length; ++i) {
            if (!internalFeedDict.hasKey(outputSymbolicTensors[i])) {
                internalFeedDict.add(outputSymbolicTensors[i], outputTensors[i], Array.isArray(outputMask) ? outputMask[0] : outputMask);
            }
            const index = outputNames.indexOf(outputSymbolicTensors[i].name);
            if (index !== -1) {
                finalOutputs[index] = outputTensors[i];
            }
        }
        if (!training) {
            // Clean up Tensors that are no longer needed.
            dispose(tensorsToDispose);
        }
    }
    // NOTE(cais): Unlike intermediate tensors, we don't discard mask
    // tensors as we go, because these tensors are sometimes passed over a
    // series of mutliple layers, i.e., not obeying the immediate input
    // relations in the graph. If this becomes a memory-usage concern,
    // we can improve this in the future.
    internalFeedDict.disposeMasks();
    return arrayFetches ? finalOutputs : finalOutputs[0];
}
/**
 * Sort the `SymbolicTensor`s topologically, for an array of fetches.
 *
 * This function calls getTopologicalSortAndRecipientCountsForOneFetch and
 * merges their results.
 *
 * @param fetch The array of fetches requested. Must be a non-empty array.
 * @param feedDict The dictionary of fed values.
 * @returns sorted: Topologically-sorted array of SymbolicTensors.
 *   recipientCounts: Recipient counts for all SymbolicTensors in `sorted`.
 */
function getTopologicalSortAndRecipientCounts(fetches, feedDict) {
    util.assert(fetches != null && fetches.length > 0, () => `Expected at least one fetch, got none`);
    let finalSorted = [];
    let finalRecipientMap = {};
    if (fetches.length === 1) {
        // Special-casing 1 fetch for efficiency.
        const out = getTopologicalSortAndRecipientCountsForOneFetch(fetches[0], feedDict);
        finalSorted = out.sorted;
        finalRecipientMap = out.recipientMap;
    }
    else {
        const visited = new Set();
        for (const fetch of fetches) {
            const { sorted, recipientMap } = getTopologicalSortAndRecipientCountsForOneFetch(fetch, feedDict);
            // Merge sorted SymbolicTensor Arrays.
            for (const symbolicTensor of sorted) {
                if (!visited.has(symbolicTensor.name)) {
                    finalSorted.push(symbolicTensor);
                    visited.add(symbolicTensor.name);
                }
            }
            // Merge recipient maps.
            for (const name in recipientMap) {
                if (finalRecipientMap[name] == null) {
                    finalRecipientMap[name] = new Set();
                }
                recipientMap[name].forEach(recipient => finalRecipientMap[name].add(recipient));
            }
        }
    }
    return {
        sorted: finalSorted,
        recipientCounts: recipientMap2Counts(finalRecipientMap)
    };
}
function recipientMap2Counts(recipientMap) {
    const recipientCounts = {};
    for (const name in recipientMap) {
        recipientCounts[name] = recipientMap[name].size;
    }
    return recipientCounts;
}
/**
 * Sort the `SymbolicTensor`s topologically, for a single fetch.
 *
 * This helper function processes the upstream SymbolicTensors of a single
 * fetch.
 *
 * @param fetch The single fetch requested.
 * @param feedDict The dictionary of fed values.
 * @returns sorted: Topologically-sorted array of SymbolicTensors.
 *   recipientMap: Recipient names for all SymbolicTensors in `sorted`.
 */
export function getTopologicalSortAndRecipientCountsForOneFetch(fetch, feedDict) {
    const visited = new Set();
    const sorted = [];
    const recipientMap = {};
    // Put keys of the feedDict into visited first, so they don't have to be
    // walked. This is needed in case where there are feeds for intermediate
    // SymbolicTensors of the graph.
    for (const key of feedDict.names()) {
        visited.add(key);
    }
    const stack = [];
    const marks = [];
    // Initial population of stack and marks.
    stack.push(fetch);
    while (stack.length > 0) {
        const top = stack[stack.length - 1];
        if (visited.has(top.name)) {
            stack.pop();
            continue;
        }
        const topIsMarked = marks[marks.length - 1] === stack.length - 1;
        if (top.inputs.length === 0 || topIsMarked) {
            // Input SymbolicTensor or all children have been visited.
            stack.pop();
            sorted.push(top);
            visited.add(top.name);
            if (topIsMarked) {
                marks.pop();
            }
        }
        else {
            // A non-input SymbolicTensor whose upstream SymbolicTensors haven't
            // been visited yet. Push them onto the stack.
            marks.push(stack.length - 1);
            for (const input of top.inputs) {
                // Increment the recipient count. Note that this needs to happen
                // regardless of whether the SymbolicTensor has been visited before.
                if (recipientMap[input.name] == null) {
                    recipientMap[input.name] = new Set();
                }
                recipientMap[input.name].add(top.name);
                if (visited.has(input.name)) {
                    continue; // Avoid repeated visits to the same SymbolicTensor.
                }
                stack.push(input);
            }
        }
    }
    return { sorted, recipientMap };
}
/**
 * Get the symbolic output tensors of the node to which a given fetch belongs.
 * @param fetch The fetched symbolic tensor.
 * @returns The Array of symbolic tensors output by the node to which `fetch`
 *   belongs.
 */
function getNodeOutputs(fetch) {
    let layerOutputs;
    if (fetch.sourceLayer.inboundNodes.length === 1) {
        layerOutputs = fetch.sourceLayer.output;
    }
    else {
        let nodeIndex = null;
        for (let i = 0; i < fetch.sourceLayer.inboundNodes.length; ++i) {
            for (const outputTensor of fetch.sourceLayer.inboundNodes[i]
                .outputTensors) {
                if (outputTensor.id === fetch.id) {
                    nodeIndex = i;
                    break;
                }
            }
        }
        layerOutputs = fetch.sourceLayer.getOutputAt(nodeIndex);
    }
    return layerOutputs;
}
//# sourceMappingURL=data:application/json;base64,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