/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from './index';
import { ALL_ENVS, describeWithFlags } from './jasmine_util';
import { complex, scalar, tensor2d } from './ops/ops';
import { inferShape } from './tensor_util_env';
import * as util from './util';
describe('Util', () => {
    it('Correctly gets size from shape', () => {
        expect(util.sizeFromShape([1, 2, 3, 4])).toEqual(24);
    });
    it('Correctly identifies scalars', () => {
        expect(util.isScalarShape([])).toBe(true);
        expect(util.isScalarShape([1, 2])).toBe(false);
        expect(util.isScalarShape([1])).toBe(false);
    });
    it('Number arrays equal', () => {
        expect(util.arraysEqual([1, 2, 3, 6], [1, 2, 3, 6])).toBe(true);
        expect(util.arraysEqual([1, 2], [1, 2, 3])).toBe(false);
        expect(util.arraysEqual([1, 2, 5], [1, 2])).toBe(false);
    });
    it('Arrays shuffle randomly', () => {
        // Create 1000 numbers ordered
        const a = Array.apply(0, { length: 1000 }).map(Number.call, Number).slice(1);
        const b = [].concat(a); // copy ES5 style
        util.shuffle(a);
        expect(a).not.toEqual(b);
        expect(a.length).toEqual(b.length);
    });
    it('Multiple arrays shuffle together', () => {
        // Create 1000 numbers ordered
        const a = Array.apply(0, { length: 1000 }).map(Number.call, Number).slice(1);
        const b = [].concat(a); // copies
        const c = [].concat(a);
        util.shuffleCombo(a, b);
        expect(a).not.toEqual(c);
        expect(a).toEqual(b);
        expect(a.length).toEqual(c.length);
    });
    it('Is integer', () => {
        expect(util.isInt(0.5)).toBe(false);
        expect(util.isInt(1)).toBe(true);
    });
    it('Size to squarish shape (perfect square)', () => {
        expect(util.sizeToSquarishShape(9)).toEqual([3, 3]);
    });
    it('Size to squarish shape (prime number)', () => {
        expect(util.sizeToSquarishShape(11)).toEqual([4, 3]);
    });
    it('Size to squarish shape (almost square)', () => {
        expect(util.sizeToSquarishShape(35)).toEqual([6, 6]);
    });
    it('Size of 1 to squarish shape', () => {
        expect(util.sizeToSquarishShape(1)).toEqual([1, 1]);
    });
    it('infer shape single number', () => {
        expect(inferShape(4)).toEqual([]);
    });
    it('infer shape 1d array', () => {
        expect(inferShape([1, 2, 5])).toEqual([3]);
    });
    it('infer shape 2d array', () => {
        expect(inferShape([[1, 2, 5], [5, 4, 1]])).toEqual([2, 3]);
    });
    it('infer shape 3d array', () => {
        const a = [[[1, 2], [2, 3], [5, 6]], [[5, 6], [4, 5], [1, 2]]];
        expect(inferShape(a)).toEqual([2, 3, 2]);
    });
    it('infer shape 4d array', () => {
        const a = [
            [[[1], [2]], [[2], [3]], [[5], [6]]], [[[5], [6]], [[4], [5]], [[1], [2]]]
        ];
        expect(inferShape(a)).toEqual([2, 3, 2, 1]);
    });
    it('infer shape of typed array', () => {
        const a = new Float32Array([1, 2, 3, 4, 5]);
        expect(inferShape(a)).toEqual([5]);
    });
    it('infer shape of clamped typed array', () => {
        const a = new Uint8ClampedArray([1, 2, 3, 4, 5]);
        expect(inferShape(a)).toEqual([5]);
    });
    it('infer shape of Uint8Array[], string tensor', () => {
        const a = [new Uint8Array([1, 2]), new Uint8Array([3, 4])];
        expect(inferShape(a, 'string')).toEqual([2]);
    });
    it('infer shape of Uint8Array[][], string tensor', () => {
        const a = [
            [new Uint8Array([1]), new Uint8Array([2])],
            [new Uint8Array([1]), new Uint8Array([2])]
        ];
        expect(inferShape(a, 'string')).toEqual([2, 2]);
    });
    it('infer shape of Uint8Array[][][], string tensor', () => {
        const a = [
            [[new Uint8Array([1, 2])], [new Uint8Array([2, 1])]],
            [[new Uint8Array([1, 2])], [new Uint8Array([2, 1])]]
        ];
        expect(inferShape(a, 'string')).toEqual([2, 2, 1]);
    });
});
describe('util.flatten', () => {
    it('nested number arrays', () => {
        expect(util.flatten([[1, 2, 3], [4, 5, 6]])).toEqual([1, 2, 3, 4, 5, 6]);
        expect(util.flatten([[[1, 2], [3, 4], [5, 6], [7, 8]]])).toEqual([
            1, 2, 3, 4, 5, 6, 7, 8
        ]);
        expect(util.flatten([1, 2, 3, 4, 5, 6])).toEqual([1, 2, 3, 4, 5, 6]);
    });
    it('nested string arrays', () => {
        expect(util.flatten([['a', 'b'], ['c', [['d']]]])).toEqual([
            'a', 'b', 'c', 'd'
        ]);
        expect(util.flatten([['a', ['b']], ['c', [['d']], 'e']])).toEqual([
            'a', 'b', 'c', 'd', 'e'
        ]);
    });
    it('mixed TypedArray and number[]', () => {
        const data = [new Float32Array([1, 2]), 3, [4, 5, new Float32Array([6, 7])]];
        expect(util.flatten(data)).toEqual([1, 2, 3, 4, 5, 6, 7]);
    });
    it('nested Uint8Arrays, skipTypedArray=true', () => {
        const data = [
            [new Uint8Array([1, 2]), new Uint8Array([3, 4])],
            [new Uint8Array([5, 6]), new Uint8Array([7, 8])]
        ];
        expect(util.flatten(data, [], true)).toEqual([
            new Uint8Array([1, 2]), new Uint8Array([3, 4]), new Uint8Array([5, 6]),
            new Uint8Array([7, 8])
        ]);
    });
});
function encodeStrings(a) {
    return a.map(s => util.encodeString(s));
}
describe('util.bytesFromStringArray', () => {
    it('count bytes after utf8 encoding', () => {
        expect(util.bytesFromStringArray(encodeStrings(['a', 'bb', 'ccc'])))
            .toBe(6);
        expect(util.bytesFromStringArray(encodeStrings(['a', 'bb', 'cccddd'])))
            .toBe(9);
        expect(util.bytesFromStringArray(encodeStrings(['даниел']))).toBe(6 * 2);
    });
});
describe('util.inferDtype', () => {
    it('a single string => string', () => {
        expect(util.inferDtype('hello')).toBe('string');
    });
    it('a single boolean => bool', () => {
        expect(util.inferDtype(true)).toBe('bool');
        expect(util.inferDtype(false)).toBe('bool');
    });
    it('a single number => float32', () => {
        expect(util.inferDtype(0)).toBe('float32');
        expect(util.inferDtype(34)).toBe('float32');
    });
    it('a list of strings => string', () => {
        // Flat.
        expect(util.inferDtype(['a', 'b', 'c'])).toBe('string');
        // Nested.
        expect(util.inferDtype([
            [['a']], [['b']], [['c']], [['d']]
        ])).toBe('string');
    });
    it('a list of bools => float32', () => {
        // Flat.
        expect(util.inferDtype([false, true, false])).toBe('bool');
        // Nested.
        expect(util.inferDtype([
            [[true]], [[false]], [[true]], [[true]]
        ])).toBe('bool');
    });
    it('a list of numbers => float32', () => {
        // Flat.
        expect(util.inferDtype([0, 1, 2])).toBe('float32');
        // Nested.
        expect(util.inferDtype([[[0]], [[1]], [[2]], [[3]]])).toBe('float32');
    });
});
describe('util.repeatedTry', () => {
    it('resolves', (doneFn) => {
        let counter = 0;
        const checkFn = () => {
            counter++;
            if (counter === 2) {
                return true;
            }
            return false;
        };
        util.repeatedTry(checkFn).then(doneFn).catch(() => {
            throw new Error('Rejected backoff.');
        });
    });
    it('rejects', (doneFn) => {
        const checkFn = () => false;
        util.repeatedTry(checkFn, () => 0, 5)
            .then(() => {
            throw new Error('Backoff resolved');
        })
            .catch(doneFn);
    });
});
describe('util.inferFromImplicitShape', () => {
    it('empty shape', () => {
        const result = util.inferFromImplicitShape([], 0);
        expect(result).toEqual([]);
    });
    it('[2, 3, 4] -> [2, 3, 4]', () => {
        const result = util.inferFromImplicitShape([2, 3, 4], 24);
        expect(result).toEqual([2, 3, 4]);
    });
    it('[2, -1, 4] -> [2, 3, 4], size=24', () => {
        const result = util.inferFromImplicitShape([2, -1, 4], 24);
        expect(result).toEqual([2, 3, 4]);
    });
    it('[-1, 3, 4] -> [2, 3, 4], size=24', () => {
        const result = util.inferFromImplicitShape([-1, 3, 4], 24);
        expect(result).toEqual([2, 3, 4]);
    });
    it('[2, 3, -1] -> [2, 3, 4], size=24', () => {
        const result = util.inferFromImplicitShape([2, 3, -1], 24);
        expect(result).toEqual([2, 3, 4]);
    });
    it('[2, -1, -1] throws error', () => {
        expect(() => util.inferFromImplicitShape([2, -1, -1], 24)).toThrowError();
    });
    it('[2, 3, -1] size=13 throws error', () => {
        expect(() => util.inferFromImplicitShape([2, 3, -1], 13)).toThrowError();
    });
    it('[2, 3, 4] size=25 (should be 24) throws error', () => {
        expect(() => util.inferFromImplicitShape([2, 3, 4], 25)).toThrowError();
    });
});
describe('util parseAxisParam', () => {
    it('axis=null returns no axes for scalar', () => {
        const axis = null;
        const shape = [];
        expect(util.parseAxisParam(axis, shape)).toEqual([]);
    });
    it('axis=null returns 0 axis for Tensor1D', () => {
        const axis = null;
        const shape = [4];
        expect(util.parseAxisParam(axis, shape)).toEqual([0]);
    });
    it('axis=null returns all axes for Tensor3D', () => {
        const axis = null;
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([0, 1, 2]);
    });
    it('axis as a single number', () => {
        const axis = 1;
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([1]);
    });
    it('axis as single negative number', () => {
        const axis = -1;
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([2]);
        const axis2 = -2;
        expect(util.parseAxisParam(axis2, shape)).toEqual([1]);
        const axis3 = -3;
        expect(util.parseAxisParam(axis3, shape)).toEqual([0]);
    });
    it('axis as list of negative numbers', () => {
        const axis = [-1, -3];
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([2, 0]);
    });
    it('axis as list of positive numbers', () => {
        const axis = [0, 2];
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([0, 2]);
    });
    it('axis as combo of positive and negative numbers', () => {
        const axis = [0, -1];
        const shape = [3, 1, 2];
        expect(util.parseAxisParam(axis, shape)).toEqual([0, 2]);
    });
    it('axis out of range throws error', () => {
        const axis = -4;
        const shape = [3, 1, 2];
        expect(() => util.parseAxisParam(axis, shape)).toThrowError();
        const axis2 = 4;
        expect(() => util.parseAxisParam(axis2, shape)).toThrowError();
    });
    it('axis a list with one number out of range throws error', () => {
        const axis = [0, 4];
        const shape = [3, 1, 2];
        expect(() => util.parseAxisParam(axis, shape)).toThrowError();
    });
    it('axis with decimal value throws error', () => {
        const axis = 0.5;
        const shape = [3, 1, 2];
        expect(() => util.parseAxisParam(axis, shape)).toThrowError();
    });
});
describe('util.squeezeShape', () => {
    it('scalar', () => {
        const { newShape, keptDims } = util.squeezeShape([]);
        expect(newShape).toEqual([]);
        expect(keptDims).toEqual([]);
    });
    it('1x1 reduced to scalar', () => {
        const { newShape, keptDims } = util.squeezeShape([1, 1]);
        expect(newShape).toEqual([]);
        expect(keptDims).toEqual([]);
    });
    it('1x3x1 reduced to [3]', () => {
        const { newShape, keptDims } = util.squeezeShape([1, 3, 1]);
        expect(newShape).toEqual([3]);
        expect(keptDims).toEqual([1]);
    });
    it('1x1x4 reduced to [4]', () => {
        const { newShape, keptDims } = util.squeezeShape([1, 1, 4]);
        expect(newShape).toEqual([4]);
        expect(keptDims).toEqual([2]);
    });
    it('2x3x4 not reduction', () => {
        const { newShape, keptDims } = util.squeezeShape([2, 3, 4]);
        expect(newShape).toEqual([2, 3, 4]);
        expect(keptDims).toEqual([0, 1, 2]);
    });
    describe('with axis', () => {
        it('should only reduce dimensions specified by axis', () => {
            const { newShape, keptDims } = util.squeezeShape([1, 1, 1, 1, 4], [1, 2]);
            expect(newShape).toEqual([1, 1, 4]);
            expect(keptDims).toEqual([0, 3, 4]);
        });
        it('should only reduce dimensions specified by negative axis', () => {
            const { newShape, keptDims } = util.squeezeShape([1, 1, 1, 1, 4], [-2, -3]);
            expect(newShape).toEqual([1, 1, 4]);
            expect(keptDims).toEqual([0, 1, 4]);
        });
        it('should only reduce dimensions specified by negative axis', () => {
            const axis = [-2, -3];
            util.squeezeShape([1, 1, 1, 1, 4], axis);
            expect(axis).toEqual([-2, -3]);
        });
        it('throws error when specified axis is not squeezable', () => {
            expect(() => util.squeezeShape([1, 1, 2, 1, 4], [1, 2])).toThrowError();
        });
        it('throws error when specified negative axis is not squeezable', () => {
            expect(() => util.squeezeShape([1, 1, 2, 1, 4], [-1, -2])).toThrowError();
        });
        it('throws error when specified axis is out of range', () => {
            expect(() => util.squeezeShape([1, 1, 2, 1, 4], [11, 22])).toThrowError();
        });
        it('throws error when specified negative axis is out of range', () => {
            expect(() => util.squeezeShape([1, 1, 2, 1, 4], [
                -11, -22
            ])).toThrowError();
        });
    });
});
describe('util.checkConversionForErrors', () => {
    it('Float32Array has NaN', () => {
        expect(() => util.checkConversionForErrors(new Float32Array([1, 2, 3, NaN, 4, 255]), 'float32'))
            .toThrowError();
    });
    it('Float32Array has Infinity', () => {
        expect(() => util.checkConversionForErrors(new Float32Array([1, 2, 3, Infinity, 4, 255]), 'float32'))
            .toThrowError();
    });
    it('Int32Array has NaN', () => {
        expect(() => util.checkConversionForErrors([1, 2, 3, 4, NaN], 'int32'))
            .toThrowError();
    });
});
describe('util.hasEncodingLoss', () => {
    it('complex64 to any', () => {
        expect(util.hasEncodingLoss('complex64', 'complex64')).toBe(false);
        expect(util.hasEncodingLoss('complex64', 'float32')).toBe(true);
        expect(util.hasEncodingLoss('complex64', 'int32')).toBe(true);
        expect(util.hasEncodingLoss('complex64', 'bool')).toBe(true);
    });
    it('any to complex64', () => {
        expect(util.hasEncodingLoss('bool', 'complex64')).toBe(false);
        expect(util.hasEncodingLoss('int32', 'complex64')).toBe(false);
        expect(util.hasEncodingLoss('float32', 'complex64')).toBe(false);
        expect(util.hasEncodingLoss('complex64', 'complex64')).toBe(false);
    });
    it('any to float32', () => {
        expect(util.hasEncodingLoss('bool', 'float32')).toBe(false);
        expect(util.hasEncodingLoss('int32', 'float32')).toBe(false);
        expect(util.hasEncodingLoss('float32', 'float32')).toBe(false);
        expect(util.hasEncodingLoss('complex64', 'float32')).toBe(true);
    });
    it('float32 to any', () => {
        expect(util.hasEncodingLoss('float32', 'float32')).toBe(false);
        expect(util.hasEncodingLoss('float32', 'int32')).toBe(true);
        expect(util.hasEncodingLoss('float32', 'bool')).toBe(true);
        expect(util.hasEncodingLoss('float32', 'complex64')).toBe(false);
    });
    it('int32 to lower', () => {
        expect(util.hasEncodingLoss('int32', 'int32')).toBe(false);
        expect(util.hasEncodingLoss('int32', 'bool')).toBe(true);
    });
    it('lower to int32', () => {
        expect(util.hasEncodingLoss('bool', 'int32')).toBe(false);
    });
    it('bool to bool', () => {
        expect(util.hasEncodingLoss('bool', 'bool')).toBe(false);
    });
});
describeWithFlags('util.toNestedArray', ALL_ENVS, () => {
    it('2 dimensions', () => {
        const a = new Float32Array([1, 2, 3, 4, 5, 6]);
        expect(util.toNestedArray([2, 3], a)).toEqual([[1, 2, 3], [4, 5, 6]]);
    });
    it('3 dimensions (2x2x3)', () => {
        const a = new Float32Array([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]);
        expect(util.toNestedArray([2, 2, 3], a)).toEqual([
            [[0, 1, 2], [3, 4, 5]], [[6, 7, 8], [9, 10, 11]]
        ]);
    });
    it('3 dimensions (3x2x2)', () => {
        const a = new Float32Array([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]);
        expect(util.toNestedArray([3, 2, 2], a)).toEqual([
            [[0, 1], [2, 3]], [[4, 5], [6, 7]], [[8, 9], [10, 11]]
        ]);
    });
    it('invalid dimension', () => {
        const a = new Float32Array([1, 2, 3]);
        expect(() => util.toNestedArray([2, 2], a)).toThrowError();
    });
    it('tensor to nested array', async () => {
        const x = tensor2d([1, 2, 3, 4], [2, 2]);
        expect(util.toNestedArray(x.shape, await x.data())).toEqual([
            [1, 2], [3, 4]
        ]);
    });
    it('scalar to nested array', async () => {
        const x = scalar(1);
        expect(util.toNestedArray(x.shape, await x.data())).toEqual(1);
    });
    it('tensor with zero shape', () => {
        const a = new Float32Array([0, 1]);
        expect(util.toNestedArray([1, 0, 2], a)).toEqual([]);
    });
});
describeWithFlags('util.toNestedArray for a complex tensor', ALL_ENVS, () => {
    it('2 dimensions', () => {
        const a = new Float32Array([1, 11, 2, 12, 3, 13, 4, 14, 5, 15, 6, 16]);
        expect(util.toNestedArray([2, 3], a, true)).toEqual([
            [1, 11, 2, 12, 3, 13], [4, 14, 5, 15, 6, 16]
        ]);
    });
    it('3 dimensions (2x2x3)', () => {
        const a = new Float32Array([
            0, 50, 1, 51, 2, 52, 3, 53, 4, 54, 5, 55,
            6, 56, 7, 57, 8, 58, 9, 59, 10, 60, 11, 61
        ]);
        expect(util.toNestedArray([2, 2, 3], a, true)).toEqual([
            [[0, 50, 1, 51, 2, 52], [3, 53, 4, 54, 5, 55]],
            [[6, 56, 7, 57, 8, 58], [9, 59, 10, 60, 11, 61]]
        ]);
    });
    it('3 dimensions (3x2x2)', () => {
        const a = new Float32Array([
            0, 50, 1, 51, 2, 52, 3, 53, 4, 54, 5, 55,
            6, 56, 7, 57, 8, 58, 9, 59, 10, 60, 11, 61
        ]);
        expect(util.toNestedArray([3, 2, 2], a, true)).toEqual([
            [[0, 50, 1, 51], [2, 52, 3, 53]], [[4, 54, 5, 55], [6, 56, 7, 57]],
            [[8, 58, 9, 59], [10, 60, 11, 61]]
        ]);
    });
    it('invalid dimension', () => {
        const a = new Float32Array([1, 11, 2, 12, 3, 13]);
        expect(() => util.toNestedArray([2, 2], a, true)).toThrowError();
    });
    it('tensor to nested array', async () => {
        const x = complex([[1, 2], [3, 4]], [[11, 12], [13, 14]]);
        expect(util.toNestedArray(x.shape, await x.data(), true)).toEqual([
            [1, 11, 2, 12], [3, 13, 4, 14]
        ]);
    });
});
describe('util.fetch', () => {
    it('should call the platform fetch', () => {
        spyOn(tf.env().platform, 'fetch').and
            .callFake(async () => ({}));
        util.fetch('test/path', { method: 'GET' });
        expect(tf.env().platform.fetch).toHaveBeenCalledWith('test/path', {
            method: 'GET'
        });
    });
});
describe('util.encodeString', () => {
    it('Encode an empty string, default encoding', () => {
        const res = util.encodeString('');
        expect(res).toEqual(new Uint8Array([]));
    });
    it('Encode an empty string, utf-8 encoding', () => {
        const res = util.encodeString('', 'utf-8');
        expect(res).toEqual(new Uint8Array([]));
    });
    it('Encode an empty string, invalid decoding', () => {
        expect(() => util.encodeString('', 'foobarbax')).toThrowError();
    });
    it('Encode cyrillic letters', () => {
        const res = util.encodeString('Kaкo стe');
        expect(res).toEqual(new Uint8Array([75, 97, 208, 186, 111, 32, 209, 129, 209, 130, 101]));
    });
    it('Encode ascii letters', () => {
        const res = util.encodeString('hello');
        expect(res).toEqual(new Uint8Array([104, 101, 108, 108, 111]));
    });
});
describe('util.decodeString', () => {
    it('decode an empty string', () => {
        const s = util.decodeString(new Uint8Array([]));
        expect(s).toEqual('');
    });
    it('decode ascii', () => {
        const s = util.decodeString(new Uint8Array([104, 101, 108, 108, 111]));
        expect(s).toEqual('hello');
    });
    it('decode cyrillic', () => {
        const s = util.decodeString(new Uint8Array([75, 97, 208, 186, 111, 32, 209, 129, 209, 130, 101]));
        expect(s).toEqual('Kaкo стe');
    });
    it('decode utf-16', () => {
        const s = util.decodeString(new Uint8Array([255, 254, 237, 139, 0, 138, 4, 89, 6, 116]), 'utf-16');
        // UTF-16 allows optional presence of byte-order-mark (BOM)
        // Construct string for '语言处理', with and without BOM
        const expected = String.fromCodePoint(0x8bed, 0x8a00, 0x5904, 0x7406);
        const expectedBOM = String.fromCodePoint(0xfeff, 0x8bed, 0x8a00, 0x5904, 0x7406);
        if (s.codePointAt(0) === 0xfeff) {
            expect(s).toEqual(expectedBOM);
        }
        else {
            expect(s).toEqual(expected);
        }
    });
    it('assert promise', () => {
        const promise = new Promise(() => { });
        expect(util.isPromise(promise)).toBeTruthy();
        const promise2 = { then: () => { } };
        expect(util.isPromise(promise2)).toBeTruthy();
        const promise3 = {};
        expect(util.isPromise(promise3)).toBeFalsy();
    });
});
//# sourceMappingURL=data:application/json;base64,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