/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from './engine';
import { env } from './environment';
import { Tensor } from './tensor';
import { assert, flatten, inferDtype, isTypedArray, toTypedArray } from './util';
export function inferShape(val, dtype) {
    let firstElem = val;
    if (isTypedArray(val)) {
        return dtype === 'string' ? [] : [val.length];
    }
    if (!Array.isArray(val)) {
        return []; // Scalar.
    }
    const shape = [];
    while (Array.isArray(firstElem) ||
        isTypedArray(firstElem) && dtype !== 'string') {
        shape.push(firstElem.length);
        firstElem = firstElem[0];
    }
    if (Array.isArray(val) &&
        env().getBool('TENSORLIKE_CHECK_SHAPE_CONSISTENCY')) {
        deepAssertShapeConsistency(val, shape, []);
    }
    return shape;
}
function deepAssertShapeConsistency(val, shape, indices) {
    indices = indices || [];
    if (!(Array.isArray(val)) && !isTypedArray(val)) {
        assert(shape.length === 0, () => `Element arr[${indices.join('][')}] is a primitive, ` +
            `but should be an array/TypedArray of ${shape[0]} elements`);
        return;
    }
    assert(shape.length > 0, () => `Element arr[${indices.join('][')}] should be a primitive, ` +
        `but is an array of ${val.length} elements`);
    assert(val.length === shape[0], () => `Element arr[${indices.join('][')}] should have ${shape[0]} ` +
        `elements, but has ${val.length} elements`);
    const subShape = shape.slice(1);
    for (let i = 0; i < val.length; ++i) {
        deepAssertShapeConsistency(val[i], subShape, indices.concat(i));
    }
}
function assertDtype(expectedDtype, actualDType, argName, functionName) {
    if (expectedDtype === 'string_or_numeric') {
        return;
    }
    if (expectedDtype == null) {
        throw new Error(`Expected dtype cannot be null.`);
    }
    if (expectedDtype !== 'numeric' && expectedDtype !== actualDType ||
        expectedDtype === 'numeric' && actualDType === 'string') {
        throw new Error(`Argument '${argName}' passed to '${functionName}' must ` +
            `be ${expectedDtype} tensor, but got ${actualDType} tensor`);
    }
}
export function convertToTensor(x, argName, functionName, parseAsDtype = 'numeric') {
    if (x instanceof Tensor) {
        assertDtype(parseAsDtype, x.dtype, argName, functionName);
        return x;
    }
    let inferredDtype = inferDtype(x);
    // If the user expects a bool/int/float, use that info to update the
    // inferredDtype when it is not a string.
    if (inferredDtype !== 'string' &&
        ['bool', 'int32', 'float32'].indexOf(parseAsDtype) >= 0) {
        inferredDtype = parseAsDtype;
    }
    assertDtype(parseAsDtype, inferredDtype, argName, functionName);
    if ((x == null) ||
        (!isTypedArray(x) && !Array.isArray(x) && typeof x !== 'number' &&
            typeof x !== 'boolean' && typeof x !== 'string')) {
        const type = x == null ? 'null' : x.constructor.name;
        throw new Error(`Argument '${argName}' passed to '${functionName}' must be a ` +
            `Tensor or TensorLike, but got '${type}'`);
    }
    const inferredShape = inferShape(x, inferredDtype);
    if (!isTypedArray(x) && !Array.isArray(x)) {
        x = [x];
    }
    const skipTypedArray = true;
    const values = inferredDtype !== 'string' ?
        toTypedArray(x, inferredDtype) :
        flatten(x, [], skipTypedArray);
    return ENGINE.makeTensor(values, inferredShape, inferredDtype);
}
export function convertToTensorArray(arg, argName, functionName, parseAsDtype = 'numeric') {
    if (!Array.isArray(arg)) {
        throw new Error(`Argument ${argName} passed to ${functionName} must be a ` +
            '`Tensor[]` or `TensorLike[]`');
    }
    const tensors = arg;
    return tensors.map((t, i) => convertToTensor(t, `${argName}[${i}]`, functionName, parseAsDtype));
}
//# sourceMappingURL=data:application/json;base64,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