/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { describeWithFlags, NODE_ENVS } from '../jasmine_util';
import * as platform_node from './platform_node';
import { PlatformNode } from './platform_node';
describeWithFlags('PlatformNode', NODE_ENVS, () => {
    it('fetch should use global.fetch if defined', async () => {
        const globalFetch = tf.env().global.fetch;
        spyOn(tf.env().global, 'fetch').and.returnValue(() => { });
        const platform = new PlatformNode();
        await platform.fetch('test/url', { method: 'GET' });
        expect(tf.env().global.fetch).toHaveBeenCalledWith('test/url', {
            method: 'GET'
        });
        tf.env().global.fetch = globalFetch;
    });
    it('fetch should use node-fetch with tf.env().global.fetch is null', async () => {
        const globalFetch = tf.env().global.fetch;
        tf.env().global.fetch = null;
        const platform = new PlatformNode();
        const savedFetch = platform_node.getSystemFetch();
        // Null out the system fetch so we force it to require node-fetch.
        platform_node.resetSystemFetch();
        const testFetch = { fetch: (url, init) => () => { } };
        // Mock the actual fetch call.
        spyOn(testFetch, 'fetch').and.returnValue(() => { });
        // Mock the import to override the real require of node-fetch.
        spyOn(platform_node.getNodeFetch, 'importFetch')
            .and.callFake(() => (url, init) => testFetch.fetch(url, init));
        await platform.fetch('test/url', { method: 'GET' });
        expect(platform_node.getNodeFetch.importFetch).toHaveBeenCalled();
        expect(testFetch.fetch).toHaveBeenCalledWith('test/url', {
            method: 'GET'
        });
        platform_node.setSystemFetch(savedFetch);
        tf.env().global.fetch = globalFetch;
    });
    it('now should use process.hrtime', async () => {
        const time = [100, 200];
        spyOn(process, 'hrtime').and.returnValue(time);
        expect(tf.env().platform.now()).toEqual(time[0] * 1000 + time[1] / 1000000);
    });
    it('encodeUTF8 single string', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('hello', 'utf-8');
        expect(bytes.length).toBe(5);
        expect(bytes).toEqual(new Uint8Array([104, 101, 108, 108, 111]));
    });
    it('encodeUTF8 two strings delimited', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('hello\x00world', 'utf-8');
        expect(bytes.length).toBe(11);
        expect(bytes).toEqual(new Uint8Array([104, 101, 108, 108, 111, 0, 119, 111, 114, 108, 100]));
    });
    it('encodeUTF8 cyrillic', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('Здраво', 'utf-8');
        expect(bytes.length).toBe(12);
        expect(bytes).toEqual(new Uint8Array([208, 151, 208, 180, 209, 128, 208, 176, 208, 178, 208, 190]));
    });
    it('decode single string', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([104, 101, 108, 108, 111]), 'utf8');
        expect(s.length).toBe(5);
        expect(s).toEqual('hello');
    });
    it('decode two strings delimited', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([104, 101, 108, 108, 111, 0, 119, 111, 114, 108, 100]), 'utf8');
        expect(s.length).toBe(11);
        expect(s).toEqual('hello\x00world');
    });
    it('decode cyrillic', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([208, 151, 208, 180, 209, 128, 208, 176, 208, 178, 208, 190]), 'utf8');
        expect(s.length).toBe(6);
        expect(s).toEqual('Здраво');
    });
});
//# sourceMappingURL=data:application/json;base64,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