/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../engine';
import { dispose, tidy } from '../globals';
import { add } from '../ops/add';
import { div } from '../ops/div';
import { mul } from '../ops/mul';
import { sqrt } from '../ops/sqrt';
import { square } from '../ops/square';
import { sub } from '../ops/sub';
import { zerosLike } from '../ops/zeros_like';
import { registerClass } from '../serialization';
import { Optimizer } from './optimizer';
/** @doclink Optimizer */
export class RMSPropOptimizer extends Optimizer {
    constructor(learningRate, decay = 0.9, momentum = 0.0, epsilon = null, centered = false) {
        super();
        this.learningRate = learningRate;
        this.decay = decay;
        this.momentum = momentum;
        this.epsilon = epsilon;
        this.accumulatedMeanSquares = [];
        this.accumulatedMoments = [];
        this.accumulatedMeanGrads = [];
        this.centered = centered;
        if (epsilon == null) {
            this.epsilon = ENGINE.backend.epsilon();
        }
        if (learningRate == null) {
            throw new Error(`learningRate for RMSPropOptimizer must be defined.`);
        }
    }
    applyGradients(variableGradients) {
        const variableNames = Array.isArray(variableGradients) ?
            variableGradients.map(item => item.name) :
            Object.keys(variableGradients);
        variableNames.forEach((name, i) => {
            const value = ENGINE.registeredVariables[name];
            const trainable = false;
            if (this.accumulatedMeanSquares[i] == null) {
                this.accumulatedMeanSquares[i] = {
                    originalName: `${name}/rms`,
                    variable: tidy(() => zerosLike(value).variable(trainable))
                };
            }
            if (this.accumulatedMoments[i] == null) {
                this.accumulatedMoments[i] = {
                    originalName: `${name}/momentum`,
                    variable: tidy(() => zerosLike(value).variable(trainable))
                };
            }
            if (this.accumulatedMeanGrads[i] == null && this.centered) {
                this.accumulatedMeanGrads[i] = {
                    originalName: `${name}/mg`,
                    variable: tidy(() => zerosLike(value).variable(trainable))
                };
            }
            const gradient = Array.isArray(variableGradients) ?
                variableGradients[i].tensor :
                variableGradients[name];
            if (gradient == null) {
                return;
            }
            const accumulatedMeanSquare = this.accumulatedMeanSquares[i].variable;
            const accumulatedMoments = this.accumulatedMoments[i].variable;
            tidy(() => {
                const newAccumulatedMeanSquare = add(mul(accumulatedMeanSquare, this.decay), mul(square(gradient), 1 - this.decay));
                if (this.centered) {
                    const accumulatedMeanGrad = this.accumulatedMeanGrads[i].variable;
                    // Centered gradient
                    const newAccumulatedMeanGrad = add(mul(accumulatedMeanGrad, this.decay), mul(gradient, 1 - this.decay));
                    const gradContribution = div(mul(gradient, this.learningRate), sqrt(sub(newAccumulatedMeanSquare, add(square(newAccumulatedMeanGrad), this.epsilon))));
                    const newAccumulatedMoments = add(mul(accumulatedMoments, this.momentum), gradContribution);
                    accumulatedMeanSquare.assign(newAccumulatedMeanSquare);
                    accumulatedMeanGrad.assign(newAccumulatedMeanGrad);
                    accumulatedMoments.assign(newAccumulatedMoments);
                    const newValue = sub(value, newAccumulatedMoments);
                    value.assign(newValue);
                }
                else {
                    // Plain gradient
                    const newAccumulatedMeanSquare = add(mul(accumulatedMeanSquare, this.decay), mul(square(gradient), 1 - this.decay));
                    const newAccumulatedMoments = add(mul(accumulatedMoments, this.momentum), div(mul(gradient, this.learningRate), sqrt(add(newAccumulatedMeanSquare, this.epsilon))));
                    accumulatedMeanSquare.assign(newAccumulatedMeanSquare);
                    accumulatedMoments.assign(newAccumulatedMoments);
                    const newValue = sub(value, newAccumulatedMoments);
                    value.assign(newValue);
                }
            });
        });
        this.incrementIterations();
    }
    dispose() {
        if (this.accumulatedMeanSquares != null) {
            dispose(this.accumulatedMeanSquares.map(v => v.variable));
        }
        if (this.accumulatedMeanGrads != null && this.centered) {
            dispose(this.accumulatedMeanGrads.map(v => v.variable));
        }
        if (this.accumulatedMoments != null) {
            dispose(this.accumulatedMoments.map(v => v.variable));
        }
    }
    async getWeights() {
        // Order matters for Python compatibility.
        const variables = [...this.accumulatedMeanSquares, ...this.accumulatedMoments];
        if (this.centered) {
            variables.push(...this.accumulatedMeanGrads);
        }
        return [await this.saveIterations()].concat(variables.map(v => ({ name: v.originalName, tensor: v.variable })));
    }
    async setWeights(weightValues) {
        weightValues = await this.extractIterations(weightValues);
        const variableCount = this.centered ? weightValues.length / 3 : weightValues.length / 2;
        const trainable = false;
        this.accumulatedMeanSquares =
            weightValues.slice(0, variableCount).map(v => ({
                originalName: v.name,
                variable: v.tensor.variable(trainable)
            }));
        this.accumulatedMoments =
            weightValues.slice(variableCount, variableCount * 2)
                .map(v => ({
                originalName: v.name,
                variable: v.tensor.variable(trainable)
            }));
        if (this.centered) {
            this.accumulatedMeanGrads =
                weightValues.slice(variableCount * 2, variableCount * 3)
                    .map(v => ({
                    originalName: v.name,
                    variable: v.tensor.variable(trainable)
                }));
        }
    }
    getConfig() {
        return {
            'learningRate': this.learningRate,
            'decay': this.decay,
            'momentum': this.momentum,
            'epsilon': this.epsilon,
            'centered': this.centered
        };
    }
    /** @nocollapse */
    static fromConfig(cls, config) {
        return new cls(config['learningRate'], config['decay'], config['momentum'], config['epsilon'], config['centered']);
    }
}
/** @nocollapse */
RMSPropOptimizer.className = 'RMSProp'; // Note: Name matters for Python compatibility.
registerClass(RMSPropOptimizer);
//# sourceMappingURL=data:application/json;base64,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