/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArrayInMeanStdRange } from './rand_util';
describeWithFlags('truncatedNormal', ALL_ENVS, () => {
    // Expect slightly higher variances for truncated values.
    const EPSILON = 0.60;
    const SEED = 2002;
    function assertTruncatedValues(values, mean, stdv) {
        const bounds = mean + stdv * 2;
        for (let i = 0; i < values.length; i++) {
            expect(Math.abs(values[i])).toBeLessThanOrEqual(bounds);
        }
    }
    it('should return a random 1D float32 array', async () => {
        const shape = [1000];
        // Ensure defaults to float32 w/o type:
        let result = tf.truncatedNormal(shape, 0, 3.5, null, SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 3.5);
        expectArrayInMeanStdRange(await result.data(), 0, 3.5, EPSILON);
        result = tf.truncatedNormal(shape, 0, 4.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 4.5);
        expectArrayInMeanStdRange(await result.data(), 0, 4.5, EPSILON);
    });
    it('should return a randon 1D int32 array', async () => {
        const shape = [1000];
        const result = tf.truncatedNormal(shape, 0, 5, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        assertTruncatedValues(await result.data(), 0, 5);
        expectArrayInMeanStdRange(await result.data(), 0, 5, EPSILON);
    });
    it('should return a 2D float32 array', async () => {
        const shape = [50, 50];
        // Ensure defaults to float32 w/o type:
        let result = tf.truncatedNormal(shape, 0, 3.5, null, SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 3.5);
        expectArrayInMeanStdRange(await result.data(), 0, 3.5, EPSILON);
        result = tf.truncatedNormal(shape, 0, 4.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 4.5);
        expectArrayInMeanStdRange(await result.data(), 0, 4.5, EPSILON);
    });
    it('should return a 2D int32 array', async () => {
        const shape = [50, 50];
        const result = tf.truncatedNormal(shape, 0, 5, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        assertTruncatedValues(await result.data(), 0, 5);
        expectArrayInMeanStdRange(await result.data(), 0, 5, EPSILON);
    });
    it('should return a 3D float32 array', async () => {
        const shape = [10, 10, 10];
        // Ensure defaults to float32 w/o type:
        let result = tf.truncatedNormal(shape, 0, 3.5, null, SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 3.5);
        expectArrayInMeanStdRange(await result.data(), 0, 3.5, EPSILON);
        result = tf.truncatedNormal(shape, 0, 4.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 4.5);
        expectArrayInMeanStdRange(await result.data(), 0, 4.5, EPSILON);
    });
    it('should return a 3D int32 array', async () => {
        const shape = [10, 10, 10];
        const result = tf.truncatedNormal(shape, 0, 5, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        assertTruncatedValues(await result.data(), 0, 5);
        expectArrayInMeanStdRange(await result.data(), 0, 5, EPSILON);
    });
    it('should return a 4D float32 array', async () => {
        const shape = [5, 5, 5, 5];
        // Ensure defaults to float32 w/o type:
        let result = tf.truncatedNormal(shape, 0, 3.5, null, SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 3.5);
        expectArrayInMeanStdRange(await result.data(), 0, 3.5, EPSILON);
        result = tf.truncatedNormal(shape, 0, 4.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        assertTruncatedValues(await result.data(), 0, 4.5);
        expectArrayInMeanStdRange(await result.data(), 0, 4.5, EPSILON);
    });
    it('should return a 4D int32 array', async () => {
        const shape = [5, 5, 5, 5];
        const result = tf.truncatedNormal(shape, 0, 5, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        assertTruncatedValues(await result.data(), 0, 5);
        expectArrayInMeanStdRange(await result.data(), 0, 5, EPSILON);
    });
});
//# sourceMappingURL=data:application/json;base64,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