/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../../engine';
import { StringNGrams } from '../../kernel_names';
import { convertToTensor } from '../../tensor_util_env';
import { op } from '../operation';
/**
 * Creates ngrams from ragged string data.
 *
 * This op accepts a ragged tensor with 1 ragged dimension containing only
 * strings and outputs a ragged tensor with 1 ragged dimension containing ngrams
 * of that string, joined along the innermost axis.
 *
 * ```js
 * const result = tf.string.stringNGrams(
 *   ['a', 'b', 'c', 'd'], tf.tensor1d([0, 2, 4], 'int32'),
 *   '|', [1, 2], 'LP', 'RP', -1, false);
 * result['nGrams'].print(); // ['a', 'b', 'LP|a', 'a|b', 'b|RP',
 *                           //  'c', 'd', 'LP|c', 'c|d', 'd|RP']
 * result['nGramsSplits'].print(); // [0, 5, 10]
 * ```
 * @param data: The values tensor of the ragged string tensor to make ngrams out
 *     of. Must be a 1D string tensor.
 * @param dataSplits: The splits tensor of the ragged string tensor to make
 *     ngrams out of.
 * @param separator: The string to append between elements of the token. Use ""
 *     for no separator.
 * @param nGramWidths: The sizes of the ngrams to create.
 * @param leftPad: The string to use to pad the left side of the ngram sequence.
 *     Only used if pad_width !== 0.
 * @param rightPad: The string to use to pad the right side of the ngram
 *     sequence. Only used if pad_width !== 0.
 * @param padWidth: The number of padding elements to add to each side of each
 *     sequence. Note that padding will never be greater than `nGramWidths`-1
 *     regardless of this value. If `padWidth`=-1 , then add max(`nGramWidths)-1
 *     elements.
 * @param preserveShortSequences: If true, then ensure that at least one ngram
 *     is generated for each input sequence. In particular, if an input sequence
 *     is shorter than min(ngramWidth) + 2*padWidth, then generate a single
 *     ngram containing the entire sequence. If false, then no ngrams are
 *     generated for these short input sequences.
 * @return A map with the following properties:
 *     - nGrams: The values tensor of the output ngrams ragged tensor.
 *     - nGramsSplits: The splits tensor of the output ngrams ragged tensor.
 *
 * @doc {heading: 'Operations', subheading: 'String'}
 */
function stringNGrams_(data, dataSplits, separator, nGramWidths, leftPad, rightPad, padWidth, preserveShortSequences) {
    const $data = convertToTensor(data, 'data', 'stringNGrams', 'string');
    if ($data.dtype !== 'string') {
        throw new Error('Data must be of datatype string');
    }
    if ($data.shape.length !== 1) {
        throw new Error(`Data must be a vector, saw: ${$data.shape}`);
    }
    const $dataSplits = convertToTensor(dataSplits, 'dataSplits', 'stringNGrams');
    if ($dataSplits.dtype !== 'int32') {
        throw new Error('Data splits must be of datatype int32');
    }
    const attrs = {
        separator,
        nGramWidths,
        leftPad,
        rightPad,
        padWidth,
        preserveShortSequences
    };
    const inputs = { data: $data, dataSplits: $dataSplits };
    const result = ENGINE.runKernel(StringNGrams, inputs, attrs);
    return { nGrams: result[0], nGramsSplits: result[1] };
}
export const stringNGrams = op({ stringNGrams_ });
//# sourceMappingURL=data:application/json;base64,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