/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../engine';
import { SparseToDense } from '../kernel_names';
import * as sparse_to_dense from '../ops/sparse_to_dense_util';
import { convertToTensor } from '../tensor_util_env';
import { op } from './operation';
/**
 * Converts a sparse representation into a dense tensor.
 *
 * Builds an array dense with shape outputShape such that:
 *
 * // If sparseIndices is scalar
 * dense[i] = (i == sparseIndices ? sparseValues : defaultValue)
 *
 * // If sparseIndices is a vector, then for each i
 * dense[sparseIndices[i]] = sparseValues[i]
 *
 * // If sparseIndices is an n by d matrix, then for each i in [0, n)
 * dense[sparseIndices[i][0], ..., sparseIndices[i][d-1]] = sparseValues[i]
 * All other values in dense are set to defaultValue. If sparseValues is a
 * scalar, all sparse indices are set to this single value.
 *
 * If indices are repeated the final value is summed over all values for those
 * indices.
 *
 * ```js
 * const indices = tf.tensor1d([4, 5, 6, 1, 2, 3], 'int32');
 * const values = tf.tensor1d([10, 11, 12, 13, 14, 15], 'float32');
 * const shape = [8];
 * tf.sparseToDense(indices, values, shape).print();
 * ```
 *
 * @param sparseIndices A 0-D, 1-D, or 2-D Tensor of type int32.
 * sparseIndices[i] contains the complete index where sparseValues[i] will be
 * placed.
 * @param sparseValues A 0-D or 1-D Tensor. Values
 * corresponding to each row of sparseIndices, or a scalar value to be used for
 * all sparse indices.
 * @param outputShape Shape of the dense output tensor. the type is inferred.
 * @param defaultValue Scalar. Value to set for indices not specified in
 * sparseIndices. Defaults to zero.
 *
 * @doc {heading: 'Operations', subheading: 'Normalization'}
 */
function sparseToDense_(sparseIndices, sparseValues, outputShape, defaultValue = 0) {
    const $sparseIndices = convertToTensor(sparseIndices, 'sparseIndices', 'sparseToDense', 'int32');
    const $sparseValues = convertToTensor(sparseValues, 'sparseValues', 'sparseToDense', 'string_or_numeric');
    const $defaultValue = convertToTensor(defaultValue, 'defaultValue', 'sparseToDense', $sparseValues.dtype);
    sparse_to_dense.validateInput($sparseIndices, $sparseValues, outputShape, $defaultValue);
    const inputs = {
        sparseIndices: $sparseIndices,
        sparseValues: $sparseValues,
        defaultValue: $defaultValue
    };
    const attrs = { outputShape };
    return ENGINE.runKernel(SparseToDense, inputs, attrs);
}
export const sparseToDense = op({ sparseToDense_ });
//# sourceMappingURL=data:application/json;base64,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