/**
 * @license
 * Copyright 2022 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('searchSorted', ALL_ENVS, () => {
    it('test1D', async () => {
        // Tests against numpy generated data.
        const NUMPY_DATA = {
            'left-float32': [
                [
                    -945.2247924804688, -921.8904418945312, -829.9115600585938,
                    -719.2261352539062, -660.3391723632812, -603.7969970703125,
                    -591.0955200195312, -373.1516418457031, -165.39039611816406,
                    -161.61097717285156, 117.37965393066406, 340.9350280761719,
                    370.4389953613281, 384.6452331542969, 601.4891357421875,
                    752.7783203125, 756.23486328125, 756.2850341796875,
                    789.2133178710938, 936.5231323242188
                ],
                [
                    -165.95599365234375, 440.64898681640625, -999.771240234375,
                    -395.3348693847656, -706.4882202148438, -815.3228149414062,
                    -627.4795532226562, -308.8785400390625, -206.46505737304688,
                    77.63346862792969
                ],
                [8, 14, 0, 7, 4, 3, 5, 8, 8, 10]
            ],
            'left-int32': [
                [
                    -961, -893, -793, -739, -706, -576, -468, -439, -424, -412,
                    -104, -16, 148, 178, 357, 399, 496, 578, 817, 977
                ],
                [-803, -157, 915, 66, 383, -368, 373, 669, -963, 500],
                [2, 10, 19, 12, 15, 10, 15, 18, 0, 17]
            ],
            'right-float32': [
                [
                    -725.0505981445312, -721.4473266601562, -669.2916259765625,
                    -460.14422607421875, -304.4682922363281, -302.20330810546875,
                    -204.64633178710938, -143.817626953125, 243.3914337158203,
                    247.34442138671875, 326.88299560546875, 451.9959716796875,
                    501.62420654296875, 501.8848571777344, 614.7825927734375,
                    766.6121826171875, 791.7724609375, 806.8038330078125,
                    855.0171508789062, 929.6801147460938
                ],
                [
                    -795.3311157226562, -171.88803100585938, 388.8003234863281,
                    -171.64146423339844, -900.0930786132812, 71.79280853271484,
                    327.58929443359375, 29.77822494506836, 889.1895141601562,
                    173.11007690429688
                ],
                [0, 7, 11, 7, 0, 8, 11, 8, 19, 8]
            ],
            'right-int32': [
                [
                    -968, -867, -751, -725, -655, -346, -285, 54, 246, 381,
                    393, 423, 507, 510, 771, 817, 846, 858, 865, 994
                ],
                [-770, 898, -100, 156, -183, -525, 806, 147, -994, 234],
                [2, 19, 7, 8, 7, 5, 15, 8, 0, 8]
            ]
        };
        for (const side of ['left', 'right']) {
            for (const dtype of ['float32', 'int32']) {
                const key = `${side}-${dtype}`;
                const [sortedSequence, values, npAns] = NUMPY_DATA[key];
                const result = tf.searchSorted(sortedSequence, values, side);
                expectArraysClose(await result.data(), npAns);
            }
        }
    });
    it('lowerBound2D', async () => {
        for (const dtype of ['float32', 'int32']) {
            const sortedSequence = tf.tensor2d([[0, 3, 9, 9, 10], [1, 2, 3, 4, 5]], undefined, dtype);
            const values = tf.tensor2d([[2, 4, 9], [0, 2, 6]], undefined, dtype);
            const correctAns = [[1, 2, 2], [0, 1, 5]];
            const result = tf.searchSorted(sortedSequence, values, 'left');
            expectArraysClose(await result.data(), correctAns);
        }
    });
    it('upperBound2D', async () => {
        for (const dtype of ['float32', 'int32']) {
            const sortedSequence = tf.tensor2d([[0, 3, 9, 9, 10], [1, 2, 3, 4, 5]], undefined, dtype);
            const values = tf.tensor2d([[2, 4, 9], [0, 2, 6]], undefined, dtype);
            const correctAns = [[1, 2, 4], [0, 2, 5]];
            const result = tf.searchSorted(sortedSequence, values, 'right');
            expectArraysClose(await result.data(), correctAns);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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