import { computeStrides, sizeFromShape } from '../util';
/**
 * Validate gather nd inputs.
 *
 * @param tensor The tensor contains the source values.
 * @param indices The tensor contains the indices to slice the source.
 *
 * @returns [resultShape, numUpdates, sliceSize, strides]
 */
export function prepareAndValidate(tensor, indices) {
    const tensorRank = tensor.shape.length;
    const indicesRank = indices.shape.length;
    if (tensorRank < 1) {
        throw new Error('tf.gatherND() expects the input to be rank 1 or higher,' +
            ` but the rank was ${tensorRank}.`);
    }
    if (indicesRank < 1) {
        throw new Error('tf.gatherND() expects the indices to be rank 1 or higher,' +
            ` but the rank was ${indicesRank}.`);
    }
    if (indices.dtype !== 'int32') {
        throw new Error('tf.gatherND() expects the indices to be int32 type,' +
            ` but the dtype was ${indices.dtype}.`);
    }
    if (indices.shape[indicesRank - 1] > tensorRank) {
        throw new Error('index innermost dimension length must be <= tensor rank; saw: ' +
            `${indices.shape[indicesRank - 1]} vs. ${tensorRank}`);
    }
    if (sizeFromShape(tensor.shape) === 0) {
        throw new Error('Requested more than 0 entries, but input is empty.' +
            ` Input shape: ${tensor.shape}.`);
    }
    const indicesShape = indices.shape;
    const sliceRank = indicesShape[indicesShape.length - 1];
    // The result shape is
    //   indices.shape[:-1] + params.shape[indices.shape[-1]:]
    let nResult = 1;
    for (let i = 0; i < indicesShape.length - 1; ++i) {
        nResult *= indicesShape[i];
    }
    const inputShape = tensor.shape;
    const resultShape = indicesShape.slice();
    resultShape.pop();
    let sliceSize = 1;
    for (let i = sliceRank; i < tensorRank; ++i) {
        sliceSize *= inputShape[i];
        resultShape.push(inputShape[i]);
    }
    const strides = [...computeStrides(tensor.shape).map(stride => stride / sliceSize),
        1].slice(0, sliceRank);
    return [resultShape, nResult, sliceSize, strides];
}
//# sourceMappingURL=data:application/json;base64,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