/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../../engine';
import { customGrad } from '../../gradients';
import { _FusedMatMul } from '../../kernel_names';
import { makeTypesMatch } from '../../tensor_util';
import { convertToTensor } from '../../tensor_util_env';
import * as util from '../../util';
import { add } from '../add';
import * as broadcast_util from '../broadcast_util';
import { applyActivation, getFusedBiasGradient, getFusedDyActivation, shouldFuse } from '../fused_util';
import { matMul as unfusedMatMul } from '../mat_mul';
import { op } from '../operation';
import { reshape } from '../reshape';
/**
 * Computes the dot product of two matrices with optional activation and bias.
 *
 * ```js
 * const a = tf.tensor2d([-1, -2], [1, 2]);
 * const b = tf.tensor2d([1, 2, 3, 4], [2, 2]);
 * const bias = tf.tensor2d([1, 2], [1, 2]);
 *
 * tf.fused.matMul({a, b, bias, activation: 'relu'}).print();
 * ```
 *
 * @param obj An object with the following properties:
 * - `a` First matrix in dot product operation.
 * - `b` Second matrix in dot product operation.
 * - `transposeA` If true, `a` is transposed before multiplication.
 * - `transposeB` If true, `b` is transposed before multiplication.
 * - `bias` Matrix to be added to the result.
 * - `activation` Name of activation kernel (defaults to `linear`).
 * - `preluActivationWeights` Tensor of prelu weights.
 * - `leakyreluAlpha` Alpha of leakyrelu.
 */
function fusedMatMul_({ a, b, transposeA = false, transposeB = false, bias, activation = 'linear', preluActivationWeights, leakyreluAlpha = 0.2, }) {
    if (shouldFuse(ENGINE.state.gradientDepth, activation) === false) {
        let result = unfusedMatMul(a, b, transposeA, transposeB);
        if (bias != null) {
            result = add(result, bias);
        }
        return applyActivation(result, activation, preluActivationWeights, leakyreluAlpha);
    }
    let $a = convertToTensor(a, 'a', 'fused matMul');
    let $b = convertToTensor(b, 'b', 'fused matMul');
    [$a, $b] = makeTypesMatch($a, $b);
    const innerShapeA = transposeA ? $a.shape[$a.rank - 2] : $a.shape[$a.rank - 1];
    const innerShapeB = transposeB ? $b.shape[$b.rank - 1] : $b.shape[$b.rank - 2];
    const outerShapeA = transposeA ? $a.shape[$a.rank - 1] : $a.shape[$a.rank - 2];
    const outerShapeB = transposeB ? $b.shape[$b.rank - 2] : $b.shape[$b.rank - 1];
    const outerDimsA = $a.shape.slice(0, -2);
    const outerDimsB = $b.shape.slice(0, -2);
    const batchDimA = util.sizeFromShape(outerDimsA);
    const batchDimB = util.sizeFromShape(outerDimsB);
    util.assert(innerShapeA === innerShapeB, () => `Error in fused matMul: inner shapes (${innerShapeA}) and (` +
        `${innerShapeB}) of Tensors with shapes ${$a.shape} and ` +
        `${$b.shape} and transposeA=${transposeA}` +
        ` and transposeB=${transposeB} must match.`);
    const outShapeOuterDims = broadcast_util.assertAndGetBroadcastShape($a.shape.slice(0, -2), $b.shape.slice(0, -2));
    const outShape = outShapeOuterDims.concat([outerShapeA, outerShapeB]);
    const a3D = transposeA ?
        reshape($a, [batchDimA, innerShapeA, outerShapeA]) :
        reshape($a, [batchDimA, outerShapeA, innerShapeA]);
    const b3D = transposeB ?
        reshape($b, [batchDimB, outerShapeB, innerShapeB]) :
        reshape($b, [batchDimB, innerShapeB, outerShapeB]);
    let $bias;
    if (bias != null) {
        $bias = convertToTensor(bias, 'bias', 'fused matMul');
        [$bias] = makeTypesMatch($bias, $a);
        broadcast_util.assertAndGetBroadcastShape(outShape, $bias.shape);
    }
    let $preluActivationWeights;
    if (preluActivationWeights != null) {
        $preluActivationWeights = convertToTensor(preluActivationWeights, 'prelu weights', 'fused matMul');
    }
    const grad = (dy, saved) => {
        const [a3D, b3D, y, $bias] = saved;
        // we reshape dy because the result of the forward is not
        // necessarily going to be a 3d tensor due to a reshape done at the end of
        // the customOp.
        const dyActivation = getFusedDyActivation(reshape(dy, y.shape), y, activation);
        let aDer;
        let bDer;
        if (!transposeA && !transposeB) {
            aDer = unfusedMatMul(dyActivation, b3D, false, true);
            bDer = unfusedMatMul(a3D, dyActivation, true, false);
        }
        else if (!transposeA && transposeB) {
            aDer = unfusedMatMul(dyActivation, b3D, false, false);
            bDer = unfusedMatMul(dyActivation, a3D, true, false);
        }
        else if (transposeA && !transposeB) {
            aDer = unfusedMatMul(b3D, dyActivation, false, true);
            bDer = unfusedMatMul(a3D, dyActivation, false, false);
        }
        else {
            aDer = unfusedMatMul(b3D, dyActivation, true, true);
            bDer = unfusedMatMul(dyActivation, a3D, true, true);
        }
        if (bias != null) {
            const biasDer = getFusedBiasGradient($bias, dyActivation);
            return [aDer, bDer, biasDer];
        }
        else {
            return [aDer, bDer];
        }
    };
    const inputs = {
        a: a3D,
        b: b3D,
        bias: $bias,
        preluActivationWeights: $preluActivationWeights
    };
    const attrs = { transposeA, transposeB, activation, leakyreluAlpha };
    // Depending on the the params passed in we will have different number of
    // inputs and thus a a different number of elements in the gradient.
    if (bias == null) {
        const customOp = customGrad((a3D, b3D, save) => {
            const res = 
            // tslint:disable-next-line: no-unnecessary-type-assertion
            ENGINE.runKernel(_FusedMatMul, inputs, attrs);
            save([a3D, b3D, res]);
            return { value: reshape(res, outShape), gradFunc: grad };
        });
        return customOp(a3D, b3D);
    }
    else {
        const customOpWithBias = customGrad((a3D, b3D, $bias, save) => {
            const res = 
            // tslint:disable-next-line: no-unnecessary-type-assertion
            ENGINE.runKernel(_FusedMatMul, inputs, attrs);
            save([a3D, b3D, res, $bias]);
            return { value: reshape(res, outShape), gradFunc: grad };
        });
        return customOpWithBias(a3D, b3D, $bias);
    }
}
export const matMul = op({ fusedMatMul_ });
//# sourceMappingURL=data:application/json;base64,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