/**
 * @license
 * Copyright 2020 Google Inc. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { BROWSER_ENVS, describeWithFlags, NODE_ENVS } from '../jasmine_util';
import { expectArraysClose, expectArraysEqual } from '../test_util';
class MockContext {
    getImageData(x, y, width, height) {
        const data = new Uint8ClampedArray(width * height * 4);
        for (let i = 0; i < data.length; ++i) {
            data[i] = i + 1;
        }
        return { data };
    }
}
class MockCanvas {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    getContext(type) {
        return new MockContext();
    }
}
describeWithFlags('fromPixels, mock canvas', NODE_ENVS, () => {
    it('accepts a canvas-like element', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = tf.browser.fromPixels(c);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 3]);
        expectArraysEqual(await t.data(), [1, 2, 3, 5, 6, 7, 9, 10, 11, 13, 14, 15]);
    });
    it('accepts a canvas-like element, numChannels=4', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = tf.browser.fromPixels(c, 4);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 4]);
        expectArraysEqual(await t.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
    it('errors when passed a non-canvas object', () => {
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels(5)).toThrowError();
    });
});
describeWithFlags('fromPixels', BROWSER_ENVS, () => {
    it('ImageData 1x1x3', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = tf.browser.fromPixels(pixels, 3);
        expectArraysEqual(await array.data(), [0, 80, 160]);
    });
    it('ImageData 1x1x4', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = tf.browser.fromPixels(pixels, 4);
        expectArraysEqual(await array.data(), [0, 80, 160, 240]);
    });
    it('ImageData 2x2x3', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = tf.browser.fromPixels(pixels, 3);
        expectArraysEqual(await array.data(), [0, 2, 4, 8, 10, 12, 16, 18, 20, 24, 26, 28]);
    });
    it('ImageData 2x2x4', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = tf.browser.fromPixels(pixels, 4);
        expectArraysClose(await array.data(), new Int32Array([0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30]));
    });
    it('fromPixels, 3 channels', async () => {
        const pixels = new ImageData(1, 2);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        pixels.data[4] = 5;
        pixels.data[5] = 6;
        pixels.data[6] = 7;
        pixels.data[7] = 255; // Not used.
        const res = tf.browser.fromPixels(pixels, 3);
        expect(res.shape).toEqual([2, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [2, 3, 4, 5, 6, 7]);
    });
    it('fromPixels, reshape, then do tf.add()', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        const a = tf.browser.fromPixels(pixels, 3).reshape([1, 1, 1, 3]);
        const res = a.add(tf.scalar(2, 'int32'));
        expect(res.shape).toEqual([1, 1, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [4, 5, 6]);
    });
    it('fromPixels + fromPixels', async () => {
        const pixelsA = new ImageData(1, 1);
        pixelsA.data[0] = 255;
        pixelsA.data[1] = 3;
        pixelsA.data[2] = 4;
        pixelsA.data[3] = 255; // Not used.
        const pixelsB = new ImageData(1, 1);
        pixelsB.data[0] = 5;
        pixelsB.data[1] = 6;
        pixelsB.data[2] = 7;
        pixelsB.data[3] = 255; // Not used.
        const a = tf.browser.fromPixels(pixelsA, 3).toFloat();
        const b = tf.browser.fromPixels(pixelsB, 3).toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixels for PixelData type', async () => {
        const dataA = new Uint8Array([255, 3, 4, 255]);
        const pixelsA = { width: 1, height: 1, data: dataA };
        const dataB = new Uint8Array([5, 6, 7, 255]);
        const pixelsB = { width: 1, height: 1, data: dataB };
        const a = tf.browser.fromPixels(pixelsA, 3).toFloat();
        const b = tf.browser.fromPixels(pixelsB, 3).toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixels for HTMLCanvasElement', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = tf.browser.fromPixels(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixels for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        const res = tf.browser.fromPixels(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixels for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,AAAAIGZ0eXBpc29tAAACAGlzb21pc28yYXZjMW1wNDEAAAAIZnJlZQAAAu1tZGF0AAACrQYF//+p3EXpvebZSLeWLNgg2SPu73gyNjQgLSBjb3JlIDE1NSByMjkwMSA3ZDBmZjIyIC0gSC4yNjQvTVBFRy00IEFWQyBjb2RlYyAtIENvcHlsZWZ0IDIwMDMtMjAxOCAtIGh0dHA6Ly93d3cudmlkZW9sYW4ub3JnL3gyNjQuaHRtbCAtIG9wdGlvbnM6IGNhYmFjPTEgcmVmPTMgZGVibG9jaz0xOjA6MCBhbmFseXNlPTB4MzoweDExMyBtZT1oZXggc3VibWU9NyBwc3k9MSBwc3lfcmQ9MS4wMDowLjAwIG1peGVkX3JlZj0xIG1lX3JhbmdlPTE2IGNocm9tYV9tZT0xIHRyZWxsaXM9MSA4eDhkY3Q9MSBjcW09MCBkZWFkem9uZT0yMSwxMSBmYXN0X3Bza2lwPTEgY2hyb21hX3FwX29mZnNldD0tMiB0aHJlYWRzPTMgbG9va2FoZWFkX3RocmVhZHM9MSBzbGljZWRfdGhyZWFkcz0wIG5yPTAgZGVjaW1hdGU9MSBpbnRlcmxhY2VkPTAgYmx1cmF5X2NvbXBhdD0wIGNvbnN0cmFpbmVkX2ludHJhPTAgYmZyYW1lcz0zIGJfcHlyYW1pZD0yIGJfYWRhcHQ9MSBiX2JpYXM9MCBkaXJlY3Q9MSB3ZWlnaHRiPTEgb3Blbl9nb3A9MCB3ZWlnaHRwPTIga2V5aW50PTI1MCBrZXlpbnRfbWluPTEgc2NlbmVjdXQ9NDAgaW50cmFfcmVmcmVzaD0wIHJjX2xvb2thaGVhZD00MCByYz1jcmYgbWJ0cmVlPTEgY3JmPTI4LjAgcWNvbXA9MC42MCBxcG1pbj0wIHFwbWF4PTY5IHFwc3RlcD00IGlwX3JhdGlvPTEuNDAgYXE9MToxLjAwAIAAAAAwZYiEAD//8m+P5OXfBeLGOfKE3xkODvFZuBflHv/+VwJIta6cbpIo4ABLoKBaYTkTAAAC7m1vb3YAAABsbXZoZAAAAAAAAAAAAAAAAAAAA+gAAAPoAAEAAAEAAAAAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAQAAAAAAAAAAAAAAAAAAQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIAAAIYdHJhawAAAFx0a2hkAAAAAwAAAAAAAAAAAAAAAQAAAAAAAAPoAAAAAAAAAAAAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAQAAAAAAAAAAAAAAAAAAQAAAAACgAAAAWgAAAAAAJGVkdHMAAAAcZWxzdAAAAAAAAAABAAAD6AAAAAAAAQAAAAABkG1kaWEAAAAgbWRoZAAAAAAAAAAAAAAAAAAAQAAAAEAAVcQAAAAAAC1oZGxyAAAAAAAAAAB2aWRlAAAAAAAAAAAAAAAAVmlkZW9IYW5kbGVyAAAAATttaW5mAAAAFHZtaGQAAAABAAAAAAAAAAAAAAAkZGluZgAAABxkcmVmAAAAAAAAAAEAAAAMdXJsIAAAAAEAAAD7c3RibAAAAJdzdHNkAAAAAAAAAAEAAACHYXZjMQAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAACgAFoASAAAAEgAAAAAAAAAAQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABj//wAAADFhdmNDAWQACv/hABhnZAAKrNlCjfkhAAADAAEAAAMAAg8SJZYBAAZo6+JLIsAAAAAYc3R0cwAAAAAAAAABAAAAAQAAQAAAAAAcc3RzYwAAAAAAAAABAAAAAQAAAAEAAAABAAAAFHN0c3oAAAAAAAAC5QAAAAEAAAAUc3RjbwAAAAAAAAABAAAAMAAAAGJ1ZHRhAAAAWm1ldGEAAAAAAAAAIWhkbHIAAAAAAAAAAG1kaXJhcHBsAAAAAAAAAAAAAAAALWlsc3QAAAAlqXRvbwAAAB1kYXRhAAAAAQAAAABMYXZmNTguMTIuMTAw';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = tf.browser.fromPixels(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    }, 30000 /* 30 seconds */);
    it('fromPixels for HTMLVideoElement throws without loadeddata', async () => {
        const video = document.createElement('video');
        video.width = 1;
        video.height = 1;
        video.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        expect(() => tf.browser.fromPixels(video)).toThrowError();
    });
    it('throws when passed a primitive number', () => {
        const msg = /pixels passed to tf.browser.fromPixels\(\) must be either/;
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels(3)).toThrowError(msg);
    });
    it('throws when passed a string', () => {
        const msg = /pixels passed to tf.browser.fromPixels\(\) must be either/;
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels('test')).toThrowError(msg);
    });
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,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';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixels(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
    if (tf.env().getBool('IS_CHROME')) {
        it('fromPixels for ImageBitmap', async () => {
            const imageDataWidth = 1;
            const imageDataHeight = 2;
            const numChannel = 3;
            const pixels = new ImageData(imageDataWidth, imageDataHeight);
            for (let i = 0; i < imageDataWidth * imageDataHeight * 4; ++i) {
                if (i % 4 === 3) {
                    pixels.data[i] = 255;
                }
                else {
                    pixels.data[i] = i;
                }
            }
            const imageBitmap = await createImageBitmap(pixels);
            const res = tf.browser.fromPixels(imageBitmap, numChannel);
            imageBitmap.close();
            expect(res.shape).toEqual([imageDataHeight, imageDataWidth, numChannel]);
            const data = await res.data();
            expect(data.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await res.data(), [0, 1, 2, 4, 5, 6]);
        });
        it('fromPixels for ImageBitmap outShape changes', async () => {
            const imageDataWidth = 2;
            const imageDataHeight = 2;
            let numChannel = 3;
            const pixels = new ImageData(imageDataWidth, imageDataHeight);
            for (let i = 0; i < imageDataWidth * imageDataHeight * 4; ++i) {
                if (i % 4 === 3) {
                    pixels.data[i] = 255;
                }
                else {
                    pixels.data[i] = i;
                }
            }
            const imageBitmap = await createImageBitmap(pixels);
            const res = tf.browser.fromPixels(imageBitmap, numChannel);
            expect(res.shape).toEqual([imageDataHeight, imageDataWidth, numChannel]);
            const data = await res.data();
            expect(data.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await res.data(), [0, 1, 2, 4, 5, 6, 8, 9, 10, 12, 13, 14]);
            // Change output shapes
            numChannel = 4;
            const resShapeChange = tf.browser.fromPixels(imageBitmap, numChannel);
            expect(resShapeChange.shape).toEqual([
                imageDataHeight, imageDataWidth, numChannel
            ]);
            const data2 = await resShapeChange.data();
            expect(data2.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await resShapeChange.data(), [0, 1, 2, 255, 4, 5, 6, 255, 8, 9, 10, 255, 12, 13, 14, 255]);
        });
    }
});
//# sourceMappingURL=data:application/json;base64,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