/**
 * @license
 * Copyright 2020 Google Inc. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { BROWSER_ENVS, describeWithFlags, NODE_ENVS } from '../jasmine_util';
import { expectArraysClose, expectArraysEqual } from '../test_util';
class MockContext {
    getImageData(x, y, width, height) {
        const data = new Uint8ClampedArray(width * height * 4);
        for (let i = 0; i < data.length; ++i) {
            data[i] = i + 1;
        }
        return { data };
    }
}
class MockCanvas {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    getContext(type) {
        return new MockContext();
    }
}
describeWithFlags('fromPixelsAsync, mock canvas', NODE_ENVS, () => {
    it('accepts a canvas-like element', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = await tf.browser.fromPixelsAsync(c);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 3]);
        expectArraysEqual(await t.data(), [1, 2, 3, 5, 6, 7, 9, 10, 11, 13, 14, 15]);
    });
    it('accepts a canvas-like element, numChannels=4', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = await tf.browser.fromPixelsAsync(c, 4);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 4]);
        expectArraysEqual(await t.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
});
// Flag 'WRAP_TO_IMAGEBITMAP' is set by customer. The default
// value is false. The cases below won't try to wrap input to
// imageBitmap.
describeWithFlags('fromPixelsAsync, ' +
    '|WRAP_TO_IMAGEBITMAP| false', BROWSER_ENVS, () => {
    it('ImageData 1x1x3', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = await tf.browser.fromPixelsAsync(pixels, 3);
        expectArraysEqual(await array.data(), [0, 80, 160]);
    });
    it('ImageData 1x1x4', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = await tf.browser.fromPixelsAsync(pixels, 4);
        expectArraysEqual(await array.data(), [0, 80, 160, 240]);
    });
    it('ImageData 2x2x3', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = await tf.browser.fromPixelsAsync(pixels, 3);
        expectArraysEqual(await array.data(), [0, 2, 4, 8, 10, 12, 16, 18, 20, 24, 26, 28]);
    });
    it('ImageData 2x2x4', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = await tf.browser.fromPixelsAsync(pixels, 4);
        expectArraysClose(await array.data(), new Int32Array([0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30]));
    });
    it('fromPixelsAsync, 3 channels', async () => {
        const pixels = new ImageData(1, 2);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        pixels.data[4] = 5;
        pixels.data[5] = 6;
        pixels.data[6] = 7;
        pixels.data[7] = 255; // Not used.
        const res = await tf.browser.fromPixelsAsync(pixels, 3);
        expect(res.shape).toEqual([2, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [2, 3, 4, 5, 6, 7]);
    });
    it('fromPixelsAsync, reshape, then do tf.add()', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        const content = await tf.browser.fromPixelsAsync(pixels, 3);
        const a = content.reshape([1, 1, 1, 3]);
        const res = a.add(tf.scalar(2, 'int32'));
        expect(res.shape).toEqual([1, 1, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [4, 5, 6]);
    });
    it('fromPixelsAsync + fromPixelsAsync', async () => {
        const pixelsA = new ImageData(1, 1);
        pixelsA.data[0] = 255;
        pixelsA.data[1] = 3;
        pixelsA.data[2] = 4;
        pixelsA.data[3] = 255; // Not used.
        const pixelsB = new ImageData(1, 1);
        pixelsB.data[0] = 5;
        pixelsB.data[1] = 6;
        pixelsB.data[2] = 7;
        pixelsB.data[3] = 255; // Not used.
        const contentA = await tf.browser.fromPixelsAsync(pixelsA, 3);
        const contentB = await tf.browser.fromPixelsAsync(pixelsB, 3);
        const a = contentA.toFloat();
        const b = contentB.toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixelsAsync for PixelData type', async () => {
        const dataA = new Uint8Array([255, 3, 4, 255]);
        const pixelsA = { width: 1, height: 1, data: dataA };
        const dataB = new Uint8Array([5, 6, 7, 255]);
        const pixelsB = { width: 1, height: 1, data: dataB };
        const contentA = await tf.browser.fromPixelsAsync(pixelsA, 3);
        const contentB = await tf.browser.fromPixelsAsync(pixelsB, 3);
        const a = contentA.toFloat();
        const b = contentB.toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixelsAsync for HTMLCanvasElement', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = await tf.browser.fromPixelsAsync(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixelsAsync for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        const res = await tf.browser.fromPixelsAsync(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixelsAsync for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,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';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = await tf.browser.fromPixelsAsync(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    }, 30000 /* 30 seconds */);
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,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';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixelsAsync(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
});
// Flag |WRAP_TO_IMAGEBITMAP| is set by customer.
// This flag helps on image, video and canvas input cases
// for WebGPU backends. We'll cover these inputs with test
// cases set 'WRAP_TO_IMAGEBITMAP' to true.
describeWithFlags('fromPixelsAsync, ' +
    '|WRAP_TO_IMAGEBITMAP| true', BROWSER_ENVS, () => {
    beforeAll(() => {
        tf.env().set('WRAP_TO_IMAGEBITMAP', true);
    });
    afterAll(() => {
        tf.env().set('WRAP_TO_IMAGEBITMAP', false);
    });
    it('fromPixelsAsync for HTMLCanvasElement ', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = await tf.browser.fromPixelsAsync(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixelsAsync for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        const res = await tf.browser.fromPixelsAsync(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixelsAsync for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,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';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = await tf.browser.fromPixelsAsync(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    });
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,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';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixelsAsync(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
});
//# sourceMappingURL=data:application/json;base64,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