/**
 * @license
 * Copyright 2022 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { norm } from './norm';
import { op } from './operation';
/**
 * Computes the euclidean norm of scalar, vectors, and matrices.
 *
 * ```js
 * const x = tf.tensor1d([1, 2, 3, 4]);
 *
 * x.euclideanNorm().print();  // or tf.euclideanNorm(x)
 * ```
 *
 * @param x The input array.
 * @param axis Optional. If axis is null (the default), the input is
 * considered a vector and a single vector norm is computed over the entire
 * set of values in the Tensor, i.e. euclideanNorm(x) is equivalent
 * to euclideanNorm(x.reshape([-1])). If axis is a integer, the input
 * is considered a batch of vectors, and axis determines the axis in x
 * over which to compute vector norms. If axis is a 2-tuple of integer it is
 * considered a batch of matrices and axis determines the axes in NDArray
 * over which to compute a matrix norm.
 * @param keepDims Optional. If true, the norm have the same dimensionality
 * as the input.
 *
 * @doc {heading: 'Operations', subheading: 'Matrices'}
 */
function euclideanNorm_(x, axis = null, keepDims = false) {
    return norm(x, 'euclidean', axis, keepDims);
}
export const euclideanNorm = op({ euclideanNorm_ });
//# sourceMappingURL=data:application/json;base64,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