/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../engine';
import { env } from '../environment';
import { FromPixels } from '../kernel_names';
import { getKernel } from '../kernel_registry';
import { Tensor } from '../tensor';
import { convertToTensor } from '../tensor_util_env';
import { cast } from './cast';
import { op } from './operation';
import { tensor3d } from './tensor3d';
let fromPixels2DContext;
/**
 * Creates a `tf.Tensor` from an image.
 *
 * ```js
 * const image = new ImageData(1, 1);
 * image.data[0] = 100;
 * image.data[1] = 150;
 * image.data[2] = 200;
 * image.data[3] = 255;
 *
 * tf.browser.fromPixels(image).print();
 * ```
 *
 * @param pixels The input image to construct the tensor from. The
 * supported image types are all 4-channel. You can also pass in an image
 * object with following attributes:
 * `{data: Uint8Array; width: number; height: number}`
 * @param numChannels The number of channels of the output tensor. A
 * numChannels value less than 4 allows you to ignore channels. Defaults to
 * 3 (ignores alpha channel of input image).
 *
 * @returns A Tensor3D with the shape `[height, width, numChannels]`.
 *
 * Note: fromPixels can be lossy in some cases, same image may result in
 * slightly different tensor values, if rendered by different rendering
 * engines. This means that results from different browsers, or even same
 * browser with CPU and GPU rendering engines can be different. See discussion
 * in details:
 * https://github.com/tensorflow/tfjs/issues/5482
 *
 * @doc {heading: 'Browser', namespace: 'browser', ignoreCI: true}
 */
function fromPixels_(pixels, numChannels = 3) {
    // Sanity checks.
    if (numChannels > 4) {
        throw new Error('Cannot construct Tensor with more than 4 channels from pixels.');
    }
    if (pixels == null) {
        throw new Error('pixels passed to tf.browser.fromPixels() can not be null');
    }
    let isPixelData = false;
    let isImageData = false;
    let isVideo = false;
    let isImage = false;
    let isCanvasLike = false;
    let isImageBitmap = false;
    if (pixels.data instanceof Uint8Array) {
        isPixelData = true;
    }
    else if (typeof (ImageData) !== 'undefined' && pixels instanceof ImageData) {
        isImageData = true;
    }
    else if (typeof (HTMLVideoElement) !== 'undefined' &&
        pixels instanceof HTMLVideoElement) {
        isVideo = true;
    }
    else if (typeof (HTMLImageElement) !== 'undefined' &&
        pixels instanceof HTMLImageElement) {
        isImage = true;
        // tslint:disable-next-line: no-any
    }
    else if (pixels.getContext != null) {
        isCanvasLike = true;
    }
    else if (typeof (ImageBitmap) !== 'undefined' && pixels instanceof ImageBitmap) {
        isImageBitmap = true;
    }
    else {
        throw new Error('pixels passed to tf.browser.fromPixels() must be either an ' +
            `HTMLVideoElement, HTMLImageElement, HTMLCanvasElement, ImageData ` +
            `in browser, or OffscreenCanvas, ImageData in webworker` +
            ` or {data: Uint32Array, width: number, height: number}, ` +
            `but was ${pixels.constructor.name}`);
    }
    if (isVideo) {
        const HAVE_CURRENT_DATA_READY_STATE = 2;
        if (isVideo &&
            pixels.readyState <
                HAVE_CURRENT_DATA_READY_STATE) {
            throw new Error('The video element has not loaded data yet. Please wait for ' +
                '`loadeddata` event on the <video> element.');
        }
    }
    // If the current backend has 'FromPixels' registered, it has a more
    // efficient way of handling pixel uploads, so we call that.
    const kernel = getKernel(FromPixels, ENGINE.backendName);
    if (kernel != null) {
        const inputs = { pixels };
        const attrs = { numChannels };
        return ENGINE.runKernel(FromPixels, inputs, attrs);
    }
    const [width, height] = isVideo ?
        [
            pixels.videoWidth,
            pixels.videoHeight
        ] :
        [pixels.width, pixels.height];
    let vals;
    if (isCanvasLike) {
        vals =
            // tslint:disable-next-line:no-any
            pixels.getContext('2d').getImageData(0, 0, width, height).data;
    }
    else if (isImageData || isPixelData) {
        vals = pixels.data;
    }
    else if (isImage || isVideo || isImageBitmap) {
        if (fromPixels2DContext == null) {
            if (typeof document === 'undefined') {
                if (typeof OffscreenCanvas !== 'undefined' &&
                    typeof OffscreenCanvasRenderingContext2D !== 'undefined') {
                    // @ts-ignore
                    fromPixels2DContext = new OffscreenCanvas(1, 1).getContext('2d');
                }
                else {
                    throw new Error('Cannot parse input in current context. ' +
                        'Reason: OffscreenCanvas Context2D rendering is not supported.');
                }
            }
            else {
                fromPixels2DContext =
                    document.createElement('canvas').getContext('2d', { willReadFrequently: true });
            }
        }
        fromPixels2DContext.canvas.width = width;
        fromPixels2DContext.canvas.height = height;
        fromPixels2DContext.drawImage(pixels, 0, 0, width, height);
        vals = fromPixels2DContext.getImageData(0, 0, width, height).data;
    }
    let values;
    if (numChannels === 4) {
        values = new Int32Array(vals);
    }
    else {
        const numPixels = width * height;
        values = new Int32Array(numPixels * numChannels);
        for (let i = 0; i < numPixels; i++) {
            for (let channel = 0; channel < numChannels; ++channel) {
                values[i * numChannels + channel] = vals[i * 4 + channel];
            }
        }
    }
    const outShape = [height, width, numChannels];
    return tensor3d(values, outShape, 'int32');
}
// Helper functions for |fromPixelsAsync| to check whether the input can
// be wrapped into imageBitmap.
function isPixelData(pixels) {
    return (pixels != null) && (pixels.data instanceof Uint8Array);
}
function isImageBitmapFullySupported() {
    return typeof window !== 'undefined' &&
        typeof (ImageBitmap) !== 'undefined' &&
        window.hasOwnProperty('createImageBitmap');
}
function isNonEmptyPixels(pixels) {
    return pixels != null && pixels.width !== 0 && pixels.height !== 0;
}
function canWrapPixelsToImageBitmap(pixels) {
    return isImageBitmapFullySupported() && !(pixels instanceof ImageBitmap) &&
        isNonEmptyPixels(pixels) && !isPixelData(pixels);
}
/**
 * Creates a `tf.Tensor` from an image in async way.
 *
 * ```js
 * const image = new ImageData(1, 1);
 * image.data[0] = 100;
 * image.data[1] = 150;
 * image.data[2] = 200;
 * image.data[3] = 255;
 *
 * (await tf.browser.fromPixelsAsync(image)).print();
 * ```
 * This API is the async version of fromPixels. The API will first
 * check |WRAP_TO_IMAGEBITMAP| flag, and try to wrap the input to
 * imageBitmap if the flag is set to true.
 *
 * @param pixels The input image to construct the tensor from. The
 * supported image types are all 4-channel. You can also pass in an image
 * object with following attributes:
 * `{data: Uint8Array; width: number; height: number}`
 * @param numChannels The number of channels of the output tensor. A
 * numChannels value less than 4 allows you to ignore channels. Defaults to
 * 3 (ignores alpha channel of input image).
 *
 * @doc {heading: 'Browser', namespace: 'browser', ignoreCI: true}
 */
export async function fromPixelsAsync(pixels, numChannels = 3) {
    let inputs = null;
    // Check whether the backend needs to wrap |pixels| to imageBitmap and
    // whether |pixels| can be wrapped to imageBitmap.
    if (env().getBool('WRAP_TO_IMAGEBITMAP') &&
        canWrapPixelsToImageBitmap(pixels)) {
        // Force the imageBitmap creation to not do any premultiply alpha
        // ops.
        let imageBitmap;
        try {
            // wrap in try-catch block, because createImageBitmap may not work
            // properly in some browsers, e.g.
            // https://bugzilla.mozilla.org/show_bug.cgi?id=1335594
            // tslint:disable-next-line: no-any
            imageBitmap = await createImageBitmap(pixels, { premultiplyAlpha: 'none' });
        }
        catch (e) {
            imageBitmap = null;
        }
        // createImageBitmap will clip the source size.
        // In some cases, the input will have larger size than its content.
        // E.g. new Image(10, 10) but with 1 x 1 content. Using
        // createImageBitmap will clip the size from 10 x 10 to 1 x 1, which
        // is not correct. We should avoid wrapping such resouce to
        // imageBitmap.
        if (imageBitmap != null && imageBitmap.width === pixels.width &&
            imageBitmap.height === pixels.height) {
            inputs = imageBitmap;
        }
        else {
            inputs = pixels;
        }
    }
    else {
        inputs = pixels;
    }
    return fromPixels_(inputs, numChannels);
}
/**
 * Draws a `tf.Tensor` of pixel values to a byte array or optionally a
 * canvas.
 *
 * When the dtype of the input is 'float32', we assume values in the range
 * [0-1]. Otherwise, when input is 'int32', we assume values in the range
 * [0-255].
 *
 * Returns a promise that resolves when the canvas has been drawn to.
 *
 * @param img A rank-2 tensor with shape `[height, width]`, or a rank-3 tensor
 * of shape `[height, width, numChannels]`. If rank-2, draws grayscale. If
 * rank-3, must have depth of 1, 3 or 4. When depth of 1, draws
 * grayscale. When depth of 3, we draw with the first three components of
 * the depth dimension corresponding to r, g, b and alpha = 1. When depth of
 * 4, all four components of the depth dimension correspond to r, g, b, a.
 * @param canvas The canvas to draw to.
 *
 * @doc {heading: 'Browser', namespace: 'browser'}
 */
export async function toPixels(img, canvas) {
    let $img = convertToTensor(img, 'img', 'toPixels');
    if (!(img instanceof Tensor)) {
        // Assume int32 if user passed a native array.
        const originalImgTensor = $img;
        $img = cast(originalImgTensor, 'int32');
        originalImgTensor.dispose();
    }
    if ($img.rank !== 2 && $img.rank !== 3) {
        throw new Error(`toPixels only supports rank 2 or 3 tensors, got rank ${$img.rank}.`);
    }
    const [height, width] = $img.shape.slice(0, 2);
    const depth = $img.rank === 2 ? 1 : $img.shape[2];
    if (depth > 4 || depth === 2) {
        throw new Error(`toPixels only supports depth of size ` +
            `1, 3 or 4 but got ${depth}`);
    }
    if ($img.dtype !== 'float32' && $img.dtype !== 'int32') {
        throw new Error(`Unsupported type for toPixels: ${$img.dtype}.` +
            ` Please use float32 or int32 tensors.`);
    }
    const data = await $img.data();
    const multiplier = $img.dtype === 'float32' ? 255 : 1;
    const bytes = new Uint8ClampedArray(width * height * 4);
    for (let i = 0; i < height * width; ++i) {
        const rgba = [0, 0, 0, 255];
        for (let d = 0; d < depth; d++) {
            const value = data[i * depth + d];
            if ($img.dtype === 'float32') {
                if (value < 0 || value > 1) {
                    throw new Error(`Tensor values for a float32 Tensor must be in the ` +
                        `range [0 - 1] but encountered ${value}.`);
                }
            }
            else if ($img.dtype === 'int32') {
                if (value < 0 || value > 255) {
                    throw new Error(`Tensor values for a int32 Tensor must be in the ` +
                        `range [0 - 255] but encountered ${value}.`);
                }
            }
            if (depth === 1) {
                rgba[0] = value * multiplier;
                rgba[1] = value * multiplier;
                rgba[2] = value * multiplier;
            }
            else {
                rgba[d] = value * multiplier;
            }
        }
        const j = i * 4;
        bytes[j + 0] = Math.round(rgba[0]);
        bytes[j + 1] = Math.round(rgba[1]);
        bytes[j + 2] = Math.round(rgba[2]);
        bytes[j + 3] = Math.round(rgba[3]);
    }
    if (canvas != null) {
        canvas.width = width;
        canvas.height = height;
        const ctx = canvas.getContext('2d');
        const imageData = new ImageData(bytes, width, height);
        ctx.putImageData(imageData, 0, 0);
    }
    if ($img !== img) {
        $img.dispose();
    }
    return bytes;
}
export const fromPixels = op({ fromPixels_ });
//# sourceMappingURL=data:application/json;base64,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