/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from './environment';
import { getGlobal } from './global_util';
import * as log from './log';
const kernelRegistry = getGlobal('kernelRegistry', () => new Map());
const gradRegistry = getGlobal('gradRegistry', () => new Map());
/**
 * Returns the kernel function (code) associated with the provided names.
 *
 * @param kernelName The official name of the kernel.
 * @param backendName The official name of the backend.
 */
export function getKernel(kernelName, backendName) {
    const key = makeKey(kernelName, backendName);
    return kernelRegistry.get(key);
}
/**
 * Returns the registered gradient info associated with the provided kernel.
 * @param kernelName The official TF kernel name.
 */
export function getGradient(kernelName) {
    return gradRegistry.get(kernelName);
}
export function getKernelsForBackend(backendName) {
    const it = kernelRegistry.entries();
    const result = [];
    while (true) {
        const { done, value } = it.next();
        if (done) {
            break;
        }
        const [key, config] = value;
        const [backend,] = key.split('_');
        if (backend === backendName) {
            result.push(config);
        }
    }
    return result;
}
/**
 * Registers the function (forward pass) for the kernel in a global registry.
 *
 * @param config A config object with the following properties:
 * - `kernelName` The official name of the kernel.
 * - `backendName` The official name of the backend.
 * - `kernelFunc` The function to run during the forward pass of the kernel.
 * - `setupFunc` Optional. Gets called once, after the backend initializes.
 * - `disposeFunc` Optional. Gets called once, right before the backend is
 * disposed.
 */
export function registerKernel(config) {
    const { kernelName, backendName } = config;
    const key = makeKey(kernelName, backendName);
    if (kernelRegistry.has(key)) {
        log.warn(`The kernel '${kernelName}' for backend ` +
            `'${backendName}' is already registered`);
    }
    kernelRegistry.set(key, config);
}
/**
 * Registers a gradient function for a given kernel in the global registry,
 * to be used during the back-propagation of that kernel.
 *
 * @param config An object with the following properties:
 * - `kernelName` The name of the kernel that the gradient function is for.
 * - `gradFunc` The function to run during back-propagation.
 */
export function registerGradient(config) {
    const { kernelName } = config;
    if (gradRegistry.has(kernelName)) {
        // TODO (yassogba) after 3.0 assess whether we need to keep this gated
        // to debug mode.
        if (env().getBool('DEBUG')) {
            log.warn(`Overriding the gradient for '${kernelName}'`);
        }
    }
    gradRegistry.set(kernelName, config);
}
/**
 * Removes the kernel function from the registry.
 *
 * @param kernelName The official name of the kernel.
 * @param backendName The official name of the backend.
 *
 */
export function unregisterKernel(kernelName, backendName) {
    const key = makeKey(kernelName, backendName);
    if (!kernelRegistry.has(key)) {
        throw new Error(`The kernel '${kernelName}' for backend ` +
            `'${backendName}' is not registered`);
    }
    kernelRegistry.delete(key);
}
/** Removes the registered gradient from the global registry. */
export function unregisterGradient(kernelName) {
    if (!gradRegistry.has(kernelName)) {
        throw new Error(`The gradient '${kernelName}' for backend is not registered`);
    }
    gradRegistry.delete(kernelName);
}
/**
 * Finds kernels that have already been registered to a backend and re-registers
 * them for a new backend. Useful for registering custom backends.
 * @param registeredBackendName Already registered backend.
 * @param newBackendName New backend.
 */
export function copyRegisteredKernels(registeredBackendName, newBackendName) {
    const kernels = getKernelsForBackend(registeredBackendName);
    kernels.forEach(kernelConfig => {
        const newKernelConfig = Object.assign({}, kernelConfig, { backendName: newBackendName });
        registerKernel(newKernelConfig);
    });
}
function makeKey(kernelName, backendName) {
    return `${backendName}_${kernelName}`;
}
//# sourceMappingURL=data:application/json;base64,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