/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
// We use the pattern below (as opposed to require('jasmine') to create the
// jasmine module in order to avoid loading node specific modules which may
// be ignored in browser environments but cannot be ignored in react-native
// due to the pre-bundling of dependencies that it must do.
// tslint:disable-next-line:no-require-imports
const jasmineRequire = require('jasmine-core/lib/jasmine-core/jasmine.js');
const jasmineCore = jasmineRequire.core(jasmineRequire);
import { KernelBackend } from './backends/backend';
import { ENGINE } from './engine';
import { env } from './environment';
import { purgeLocalStorageArtifacts } from './io/local_storage';
import { isPromise } from './util_base';
Error.stackTraceLimit = Infinity;
jasmineCore.DEFAULT_TIMEOUT_INTERVAL = 20000;
export const NODE_ENVS = {
    predicate: () => env().platformName === 'node'
};
export const CHROME_ENVS = {
    flags: { 'IS_CHROME': true }
};
export const BROWSER_ENVS = {
    predicate: () => env().platformName === 'browser'
};
export const SYNC_BACKEND_ENVS = {
    predicate: (testEnv) => testEnv.isDataSync === true
};
export const HAS_WORKER = {
    predicate: () => typeof (Worker) !== 'undefined' &&
        typeof (Blob) !== 'undefined' && typeof (URL) !== 'undefined'
};
export const HAS_NODE_WORKER = {
    predicate: () => {
        let hasWorker = true;
        try {
            require.resolve('worker_threads');
        }
        catch (_a) {
            hasWorker = false;
        }
        return typeof (process) !== 'undefined' && hasWorker;
    }
};
export const ALL_ENVS = {};
// Tests whether the current environment satisfies the set of constraints.
export function envSatisfiesConstraints(env, testEnv, constraints) {
    if (constraints == null) {
        return true;
    }
    if (constraints.flags != null) {
        for (const flagName in constraints.flags) {
            const flagValue = constraints.flags[flagName];
            if (env.get(flagName) !== flagValue) {
                return false;
            }
        }
    }
    if (constraints.predicate != null && !constraints.predicate(testEnv)) {
        return false;
    }
    return true;
}
/**
 * Add test filtering logic to Jasmine's specFilter hook.
 *
 * @param testFilters Used for include a test suite, with the ability
 *     to selectively exclude some of the tests.
 *     Either `include` or `startsWith` must exist for a `TestFilter`.
 *     Tests that have the substrings specified by the include or startsWith
 *     will be included in the test run, unless one of the substrings specified
 *     by `excludes` appears in the name.
 * @param customInclude Function to programatically include a test.
 *     If this function returns true, a test will immediately run. Otherwise,
 *     `testFilters` is used for fine-grained filtering.
 *
 * If a test is not handled by `testFilters` or `customInclude`, the test will
 * be excluded in the test run.
 */
export function setupTestFilters(testFilters, customInclude) {
    const env = jasmine.getEnv();
    // Account for --grep flag passed to karma by saving the existing specFilter.
    const config = env.configuration();
    const grepFilter = config.specFilter;
    /**
     * Filter method that returns boolean, if a given test should run or be
     * ignored based on its name. The exclude list has priority over the
     * include list. Thus, if a test matches both the exclude and the include
     * list, it will be exluded.
     */
    // tslint:disable-next-line: no-any
    const specFilter = (spec) => {
        // Filter out tests if the --grep flag is passed.
        if (!grepFilter(spec)) {
            return false;
        }
        const name = spec.getFullName();
        if (customInclude(name)) {
            return true;
        }
        // Include tests of a test suite unless tests are in excludes list.
        for (let i = 0; i < testFilters.length; ++i) {
            const testFilter = testFilters[i];
            if ((testFilter.include != null &&
                name.indexOf(testFilter.include) > -1) ||
                (testFilter.startsWith != null &&
                    name.startsWith(testFilter.startsWith))) {
                if (testFilter.excludes != null) {
                    for (let j = 0; j < testFilter.excludes.length; j++) {
                        if (name.indexOf(testFilter.excludes[j]) > -1) {
                            return false;
                        }
                    }
                }
                return true;
            }
        }
        // Otherwise ignore the test.
        return false;
    };
    env.configure(Object.assign({}, config, { specFilter }));
}
export function parseTestEnvFromKarmaFlags(args, registeredTestEnvs) {
    let flags;
    let testEnvName;
    args.forEach((arg, i) => {
        if (arg === '--flags') {
            flags = JSON.parse(args[i + 1]);
        }
        else if (arg === '--testEnv') {
            testEnvName = args[i + 1];
        }
    });
    const testEnvNames = registeredTestEnvs.map(env => env.name).join(', ');
    if (flags != null && testEnvName == null) {
        throw new Error('--testEnv flag is required when --flags is present. ' +
            `Available values are [${testEnvNames}].`);
    }
    if (testEnvName == null) {
        return null;
    }
    let testEnv;
    registeredTestEnvs.forEach(env => {
        if (env.name === testEnvName) {
            testEnv = env;
        }
    });
    if (testEnv == null) {
        throw new Error(`Test environment with name ${testEnvName} not ` +
            `found. Available test environment names are ` +
            `${testEnvNames}`);
    }
    if (flags != null) {
        testEnv.flags = flags;
    }
    return testEnv;
}
export function describeWithFlags(name, constraints, tests) {
    if (TEST_ENVS.length === 0) {
        throw new Error(`Found no test environments. This is likely due to test environment ` +
            `registries never being imported or test environment registries ` +
            `being registered too late.`);
    }
    TEST_ENVS.forEach(testEnv => {
        env().setFlags(testEnv.flags);
        if (envSatisfiesConstraints(env(), testEnv, constraints)) {
            const testName = name + ' ' + testEnv.name + ' ' + JSON.stringify(testEnv.flags || {});
            executeTests(testName, tests, testEnv);
        }
    });
}
export const TEST_ENVS = [];
// Whether a call to setTestEnvs has been called so we turn off
// registration. This allows command line overriding or programmatic
// overriding of the default registrations.
let testEnvSet = false;
export function setTestEnvs(testEnvs) {
    testEnvSet = true;
    TEST_ENVS.length = 0;
    TEST_ENVS.push(...testEnvs);
}
export function registerTestEnv(testEnv) {
    // When using an explicit call to setTestEnvs, turn off registration of
    // test environments because the explicit call will set the test
    // environments.
    if (testEnvSet) {
        return;
    }
    TEST_ENVS.push(testEnv);
}
function executeTests(testName, tests, testEnv) {
    describe(testName, () => {
        beforeAll(async () => {
            ENGINE.reset();
            if (testEnv.flags != null) {
                env().setFlags(testEnv.flags);
            }
            env().set('IS_TEST', true);
            // Await setting the new backend since it can have async init.
            await ENGINE.setBackend(testEnv.backendName);
        });
        beforeEach(() => {
            ENGINE.startScope();
        });
        afterEach(() => {
            ENGINE.endScope();
            ENGINE.disposeVariables();
        });
        afterAll(() => {
            ENGINE.reset();
        });
        tests(testEnv);
    });
}
export class TestKernelBackend extends KernelBackend {
    dispose() { }
}
let lock = Promise.resolve();
/**
 * Wraps a Jasmine spec's test function so it is run exclusively to others that
 * use runWithLock.
 *
 * @param spec The function that runs the spec. Must return a promise or call
 *     `done()`.
 *
 */
export function runWithLock(spec) {
    return () => {
        lock = lock.then(async () => {
            let done;
            const donePromise = new Promise((resolve, reject) => {
                done = (() => {
                    resolve();
                });
                done.fail = (message) => {
                    reject(message);
                };
            });
            purgeLocalStorageArtifacts();
            const result = spec(done);
            if (isPromise(result)) {
                await result;
            }
            else {
                await donePromise;
            }
        });
        return lock;
    };
}
//# sourceMappingURL=data:application/json;base64,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