/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { GatherV2 } from '../kernel_names';
import { getUndoAxesPermutation } from '../ops/axis_util';
import { reshape } from '../ops/reshape';
import { transpose } from '../ops/transpose';
import { unsortedSegmentSum } from '../ops/unsorted_segment_sum';
import { parseAxisParam } from '../util';
export const gatherGradConfig = {
    kernelName: GatherV2,
    inputsToSave: ['x', 'indices'],
    gradFunc: (dy, saved, attrs) => {
        const [x, indices] = saved;
        const { axis } = attrs;
        const parsedAxis = parseAxisParam(axis, x.shape)[0];
        const derX = () => {
            const paramsShape = x.shape;
            const indicesSize = indices.size;
            const outerShape = paramsShape.slice(0, parsedAxis);
            const outerDims = outerShape.length;
            const innerShape = paramsShape.slice(axis, paramsShape.length).slice(1);
            const innerDims = innerShape.length;
            const outerAxesIndices = arrayRange(0, outerDims);
            const innerAxesIndices = arrayRange(outerDims + 1, outerDims + 1 + innerDims);
            const valuesShape = arrayConcat([outerShape, [indicesSize], innerShape]);
            const values = reshape(dy, valuesShape);
            const reshapedIndices = reshape(indices, [indicesSize]);
            const transposeDims = arrayConcat([[outerDims], outerAxesIndices, innerAxesIndices]);
            const valuesTranspose = transpose(values, transposeDims);
            let paramsGrad = unsortedSegmentSum(valuesTranspose, reshapedIndices, x.shape[parsedAxis]);
            const invertTransposeDims = getUndoAxesPermutation(transposeDims);
            paramsGrad = transpose(paramsGrad, invertTransposeDims);
            return paramsGrad;
        };
        return { x: derX, indices: () => indices };
    }
};
function arrayRange(start, stop) {
    const result = [];
    for (let i = start; i < stop; ++i) {
        result.push(i);
    }
    return result;
}
function arrayConcat(arrays) {
    const result = [];
    for (let i = 0; i < arrays.length; ++i) {
        for (let j = 0; j < arrays[i].length; ++j) {
            result.push(arrays[i][j]);
        }
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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