/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as environment from './environment';
import { Environment } from './environment';
describe('initializes flags from the url', () => {
    // Silence console.warns for these tests.
    beforeAll(() => spyOn(console, 'warn').and.returnValue(null));
    it('no overrides one registered flag', () => {
        const global = { location: { search: '' } };
        const env = new Environment(global);
        spyOn(env, 'getQueryParams').and.returnValue({});
        env.registerFlag('FLAG1', () => false);
        expect(env.get('FLAG1')).toBe(false);
    });
    it('one unregistered flag', () => {
        const global = { location: { search: '' } };
        const env = new Environment(global);
        spyOn(env, 'getQueryParams').and.returnValue({
            'tfjsflags': 'FLAG1:true'
        });
        expect(env.features).toEqual({});
    });
    it('one registered flag true', () => {
        const global = { location: { search: '?tfjsflags=FLAG1:true' } };
        const env = new Environment(global);
        env.registerFlag('FLAG1', () => false);
        expect(env.get('FLAG1')).toBe(true);
    });
    it('one registered flag false', () => {
        const global = { location: { search: '?tfjsflags=FLAG1:false' } };
        const env = new Environment(global);
        env.registerFlag('FLAG1', () => true);
        expect(env.get('FLAG1')).toBe(false);
    });
    it('two registered flags', () => {
        const global = { location: { search: '?tfjsflags=FLAG1:true,FLAG2:200' } };
        const env = new Environment(global);
        env.registerFlag('FLAG1', () => false);
        env.registerFlag('FLAG2', () => 100);
        expect(env.get('FLAG1')).toBe(true);
        expect(env.get('FLAG2')).toBe(200);
    });
});
describe('flag registration and evaluation', () => {
    it('one flag registered', () => {
        const env = new Environment({});
        const evalObject = { eval: () => true };
        const spy = spyOn(evalObject, 'eval').and.callThrough();
        env.registerFlag('FLAG1', () => evalObject.eval());
        expect(env.get('FLAG1')).toBe(true);
        expect(spy.calls.count()).toBe(1);
        // Multiple calls to get do not call the evaluation function again.
        expect(env.get('FLAG1')).toBe(true);
        expect(spy.calls.count()).toBe(1);
    });
    it('multiple flags registered', () => {
        const env = new Environment({});
        const evalObject = { eval1: () => true, eval2: () => 100 };
        const spy1 = spyOn(evalObject, 'eval1').and.callThrough();
        const spy2 = spyOn(evalObject, 'eval2').and.callThrough();
        env.registerFlag('FLAG1', () => evalObject.eval1());
        env.registerFlag('FLAG2', () => evalObject.eval2());
        expect(env.get('FLAG1')).toBe(true);
        expect(spy1.calls.count()).toBe(1);
        expect(spy2.calls.count()).toBe(0);
        expect(env.get('FLAG2')).toBe(100);
        expect(spy1.calls.count()).toBe(1);
        expect(spy2.calls.count()).toBe(1);
        // Multiple calls to get do not call the evaluation function again.
        expect(env.get('FLAG1')).toBe(true);
        expect(env.get('FLAG2')).toBe(100);
        expect(spy1.calls.count()).toBe(1);
        expect(spy2.calls.count()).toBe(1);
    });
    it('setting overrides value', () => {
        const env = new Environment({});
        const evalObject = { eval: () => true };
        const spy = spyOn(evalObject, 'eval').and.callThrough();
        env.registerFlag('FLAG1', () => evalObject.eval());
        expect(env.get('FLAG1')).toBe(true);
        expect(spy.calls.count()).toBe(1);
        env.set('FLAG1', false);
        expect(env.get('FLAG1')).toBe(false);
        expect(spy.calls.count()).toBe(1);
    });
    it('set hook is called', () => {
        const env = new Environment({});
        const evalObject = { eval: () => true, setHook: () => true };
        const evalSpy = spyOn(evalObject, 'eval').and.callThrough();
        const setHookSpy = spyOn(evalObject, 'setHook').and.callThrough();
        env.registerFlag('FLAG1', () => evalObject.eval(), () => evalObject.setHook());
        expect(env.get('FLAG1')).toBe(true);
        expect(evalSpy.calls.count()).toBe(1);
        expect(setHookSpy.calls.count()).toBe(0);
        env.set('FLAG1', false);
        expect(env.get('FLAG1')).toBe(false);
        expect(evalSpy.calls.count()).toBe(1);
        expect(setHookSpy.calls.count()).toBe(1);
    });
});
describe('environment.getQueryParams', () => {
    it('basic', () => {
        expect(environment.getQueryParams('?a=1&b=hi&f=animal'))
            .toEqual({ 'a': '1', 'b': 'hi', 'f': 'animal' });
    });
});
//# sourceMappingURL=data:application/json;base64,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