/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { isPromise } from './util_base';
// Expects flags from URL in the format ?tfjsflags=FLAG1:1,FLAG2:true.
const TENSORFLOWJS_FLAGS_PREFIX = 'tfjsflags';
/**
 * The environment contains evaluated flags as well as the registered platform.
 * This is always used as a global singleton and can be retrieved with
 * `tf.env()`.
 *
 * @doc {heading: 'Environment'}
 */
export class Environment {
    // tslint:disable-next-line: no-any
    constructor(global) {
        this.global = global;
        this.flags = {};
        this.flagRegistry = {};
        this.urlFlags = {};
        // Jasmine spies on this in 'environment_test.ts'
        this.getQueryParams = getQueryParams;
        this.populateURLFlags();
    }
    setPlatform(platformName, platform) {
        if (this.platform != null) {
            if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                console.warn(`Platform ${this.platformName} has already been set. ` +
                    `Overwriting the platform with ${platformName}.`);
            }
        }
        this.platformName = platformName;
        this.platform = platform;
    }
    registerFlag(flagName, evaluationFn, setHook) {
        this.flagRegistry[flagName] = { evaluationFn, setHook };
        // Override the flag value from the URL. This has to happen here because
        // the environment is initialized before flags get registered.
        if (this.urlFlags[flagName] != null) {
            const flagValue = this.urlFlags[flagName];
            if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                console.warn(`Setting feature override from URL ${flagName}: ${flagValue}.`);
            }
            this.set(flagName, flagValue);
        }
    }
    async getAsync(flagName) {
        if (flagName in this.flags) {
            return this.flags[flagName];
        }
        this.flags[flagName] = await this.evaluateFlag(flagName);
        return this.flags[flagName];
    }
    get(flagName) {
        if (flagName in this.flags) {
            return this.flags[flagName];
        }
        const flagValue = this.evaluateFlag(flagName);
        if (isPromise(flagValue)) {
            throw new Error(`Flag ${flagName} cannot be synchronously evaluated. ` +
                `Please use getAsync() instead.`);
        }
        this.flags[flagName] = flagValue;
        return this.flags[flagName];
    }
    getNumber(flagName) {
        return this.get(flagName);
    }
    getBool(flagName) {
        return this.get(flagName);
    }
    getFlags() {
        return this.flags;
    }
    // For backwards compatibility.
    get features() {
        return this.flags;
    }
    set(flagName, value) {
        if (this.flagRegistry[flagName] == null) {
            throw new Error(`Cannot set flag ${flagName} as it has not been registered.`);
        }
        this.flags[flagName] = value;
        if (this.flagRegistry[flagName].setHook != null) {
            this.flagRegistry[flagName].setHook(value);
        }
    }
    evaluateFlag(flagName) {
        if (this.flagRegistry[flagName] == null) {
            throw new Error(`Cannot evaluate flag '${flagName}': no evaluation function found.`);
        }
        return this.flagRegistry[flagName].evaluationFn();
    }
    setFlags(flags) {
        this.flags = Object.assign({}, flags);
    }
    reset() {
        this.flags = {};
        this.urlFlags = {};
        this.populateURLFlags();
    }
    populateURLFlags() {
        if (typeof this.global === 'undefined' ||
            typeof this.global.location === 'undefined' ||
            typeof this.global.location.search === 'undefined') {
            return;
        }
        const urlParams = this.getQueryParams(this.global.location.search);
        if (TENSORFLOWJS_FLAGS_PREFIX in urlParams) {
            const keyValues = urlParams[TENSORFLOWJS_FLAGS_PREFIX].split(',');
            keyValues.forEach(keyValue => {
                const [key, value] = keyValue.split(':');
                this.urlFlags[key] = parseValue(key, value);
            });
        }
    }
}
export function getQueryParams(queryString) {
    const params = {};
    queryString.replace(/[?&]([^=?&]+)(?:=([^&]*))?/g, (s, ...t) => {
        decodeParam(params, t[0], t[1]);
        return t.join('=');
    });
    return params;
}
function decodeParam(params, name, value) {
    params[decodeURIComponent(name)] = decodeURIComponent(value || '');
}
function parseValue(flagName, value) {
    value = value.toLowerCase();
    if (value === 'true' || value === 'false') {
        return value === 'true';
    }
    else if (`${+value}` === value) {
        return +value;
    }
    throw new Error(`Could not parse value flag value ${value} for flag ${flagName}.`);
}
/**
 * Returns the current environment (a global singleton).
 *
 * The environment object contains the evaluated feature values as well as the
 * active platform.
 *
 * @doc {heading: 'Environment'}
 */
export function env() {
    return ENV;
}
export let ENV = null;
export function setEnvironmentGlobal(environment) {
    ENV = environment;
}
//# sourceMappingURL=data:application/json;base64,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