/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from './index';
import { ALL_ENVS, describeWithFlags, SYNC_BACKEND_ENVS } from './jasmine_util';
import { convertToTensor } from './tensor_util_env';
import { expectArraysClose } from './test_util';
describeWithFlags('debug on', SYNC_BACKEND_ENVS, () => {
    beforeAll(() => {
        // Silence debug warnings.
        spyOn(console, 'warn');
        tf.enableDebugMode();
    });
    it('debug mode does not error when no nans', async () => {
        const a = tf.tensor1d([2, -1, 0, 3]);
        const res = tf.relu(a);
        expectArraysClose(await res.data(), [2, 0, 0, 3]);
    });
    it('debug mode errors when nans in tensor construction, float32', () => {
        const a = () => tf.tensor1d([2, NaN], 'float32');
        expect(a).toThrowError();
    });
    it('debug mode errors when nans in tensor construction, int32', () => {
        const a = () => tf.tensor1d([2, NaN], 'int32');
        expect(a).toThrowError();
    });
    it('debug mode errors when Infinity in tensor construction', () => {
        const a = () => tf.tensor1d([2, Infinity], 'float32');
        expect(a).toThrowError();
    });
    it('debug mode errors when nans in tensor created from TypedArray', () => {
        const a = () => tf.tensor1d(new Float32Array([1, 2, NaN]), 'float32');
        expect(a).toThrowError();
    });
    it('debug mode errors when infinities in op output', async () => {
        const a = tf.tensor1d([1, 2, 3, 4]);
        const b = tf.tensor1d([2, -1, 0, 3]);
        const c = async () => {
            const result = a.div(b);
            // Must await result so we know exception would have happened by the
            // time we call `expect`.
            await result.data();
        };
        await c();
        expect(console.warn).toHaveBeenCalled();
    });
    it('debug mode errors when nans in op output', async () => {
        const a = tf.tensor1d([-1, 2]);
        const b = tf.tensor1d([0.5, 1]);
        const c = async () => {
            const result = a.pow(b);
            await result.data();
        };
        await c();
        expect(console.warn).toHaveBeenCalled();
    });
    it('debug mode errors when nans in oneHot op (tensorlike), int32', () => {
        const f = () => tf.oneHot([2, NaN], 3);
        expect(f).toThrowError();
    });
    it('debug mode errors when nan in convertToTensor, int32', () => {
        const a = () => convertToTensor(NaN, 'a', 'test', 'int32');
        expect(a).toThrowError();
    });
    it('debug mode errors when nan in convertToTensor array input, int32', () => {
        const a = () => convertToTensor([NaN], 'a', 'test', 'int32');
        expect(a).toThrowError();
    });
    // tslint:disable-next-line: ban
    xit('A x B', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.matMul(a, b);
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, -3, 20]);
    });
});
describeWithFlags('debug off', ALL_ENVS, () => {
    beforeAll(() => {
        tf.env().set('DEBUG', false);
    });
    it('no errors where there are nans, and debug mode is disabled', async () => {
        const a = tf.tensor1d([2, NaN]);
        const res = tf.relu(a);
        expectArraysClose(await res.data(), [2, NaN]);
    });
});
//# sourceMappingURL=data:application/json;base64,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