/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tfc from '@tensorflow/tfjs-core';
import { NodeValueImpl } from './custom_op/node_value_impl';
import { getRegisteredOp } from './custom_op/register';
import * as arithmetic from './executors/arithmetic_executor';
import * as basicMath from './executors/basic_math_executor';
import * as control from './executors/control_executor';
import * as convolution from './executors/convolution_executor';
import * as creation from './executors/creation_executor';
import * as dynamic from './executors/dynamic_executor';
import * as evaluation from './executors/evaluation_executor';
import * as graph from './executors/graph_executor';
import * as hashTable from './executors/hash_table_executor';
import * as image from './executors/image_executor';
import * as logical from './executors/logical_executor';
import * as matrices from './executors/matrices_executor';
import * as normalization from './executors/normalization_executor';
import * as reduction from './executors/reduction_executor';
import * as sliceJoin from './executors/slice_join_executor';
import * as sparse from './executors/sparse_executor';
import * as spectral from './executors/spectral_executor';
import * as string from './executors/string_executor';
import * as transformation from './executors/transformation_executor';
/**
 * Executes the op defined by the node object.
 * @param node
 * @param tensorMap contains tensors for executed nodes and weights
 * @param context contains tensors and information for running the current node.
 * @param resourceManager Optional. Contains global resources of the model.
 */
export function executeOp(node, tensorMap, context, resourceManager, tidy = tfc.tidy) {
    const value = ((node, tensorMap, context) => {
        switch (node.category) {
            case 'arithmetic':
                return tidy(() => arithmetic.executeOp(node, tensorMap, context));
            case 'basic_math':
                return tidy(() => basicMath.executeOp(node, tensorMap, context));
            case 'control':
                return control.executeOp(node, tensorMap, context);
            case 'convolution':
                return tidy(() => convolution.executeOp(node, tensorMap, context));
            case 'creation':
                return tidy(() => creation.executeOp(node, tensorMap, context));
            case 'dynamic':
                return dynamic.executeOp(node, tensorMap, context);
            case 'evaluation':
                return tidy(() => evaluation.executeOp(node, tensorMap, context));
            case 'image':
                return tidy(() => image.executeOp(node, tensorMap, context));
            case 'graph':
                return tidy(() => graph.executeOp(node, tensorMap, context));
            case 'logical':
                return tidy(() => logical.executeOp(node, tensorMap, context));
            case 'matrices':
                return tidy(() => matrices.executeOp(node, tensorMap, context));
            case 'normalization':
                return tidy(() => normalization.executeOp(node, tensorMap, context));
            case 'reduction':
                return tidy(() => reduction.executeOp(node, tensorMap, context));
            case 'slice_join':
                return tidy(() => sliceJoin.executeOp(node, tensorMap, context));
            case 'sparse':
                return tidy(() => sparse.executeOp(node, tensorMap, context));
            case 'spectral':
                return tidy(() => spectral.executeOp(node, tensorMap, context));
            case 'string':
                return tidy(() => string.executeOp(node, tensorMap, context));
            case 'transformation':
                return tidy(() => transformation.executeOp(node, tensorMap, context));
            case 'hash_table':
                return hashTable.executeOp(node, tensorMap, context, resourceManager);
            case 'custom':
                const opMapper = getRegisteredOp(node.op);
                if (opMapper && opMapper.customExecutor) {
                    return opMapper.customExecutor(new NodeValueImpl(node, tensorMap, context));
                }
                else {
                    throw TypeError(`Custom op ${node.op} is not registered.`);
                }
            default:
                throw TypeError(`Unknown op '${node.op}'. File an issue at ` +
                    `https://github.com/tensorflow/tfjs/issues so we can add it` +
                    `, or register a custom execution with tf.registerOp()`);
        }
    })(node, tensorMap, context);
    if (tfc.util.isPromise(value)) {
        return value.then((data) => [].concat(data));
    }
    return [].concat(value);
}
//# sourceMappingURL=data:application/json;base64,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