/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { HashTable } from '../../executor/hash_table';
import { getParamValue } from './utils';
export const executeOp = async (node, tensorMap, context, resourceManager) => {
    switch (node.op) {
        case 'HashTable':
        case 'HashTableV2': {
            const keyDType = getParamValue('keyDType', node, tensorMap, context);
            const valueDType = getParamValue('valueDType', node, tensorMap, context);
            const hashTable = new HashTable(keyDType, valueDType);
            resourceManager.addHashTable(node.name, hashTable);
            return [hashTable.handle];
        }
        case 'LookupTableImport':
        case 'LookupTableImportV2': {
            const handle = getParamValue('tableHandle', node, tensorMap, context, resourceManager);
            const keys = getParamValue('keys', node, tensorMap, context);
            const values = getParamValue('values', node, tensorMap, context);
            const hashTable = resourceManager.getHashTableById(handle.id);
            return [await hashTable.import(keys, values)];
        }
        case 'LookupTableFind':
        case 'LookupTableFindV2': {
            const handle = getParamValue('tableHandle', node, tensorMap, context, resourceManager);
            const keys = getParamValue('keys', node, tensorMap, context);
            const defaultValue = getParamValue('defaultValue', node, tensorMap, context);
            const hashTable = resourceManager.getHashTableById(handle.id);
            return [await hashTable.find(keys, defaultValue)];
        }
        case 'LookupTableSize':
        case 'LookupTableSizeV2': {
            const handle = getParamValue('tableHandle', node, tensorMap, context, resourceManager);
            const hashTable = resourceManager.getHashTableById(handle.id);
            return [hashTable.tensorSize()];
        }
        default:
            throw TypeError(`Node type ${node.op} is not implemented`);
    }
};
export const CATEGORY = 'hash_table';
//# sourceMappingURL=data:application/json;base64,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