/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { io, Tensor, util } from '@tensorflow/tfjs-core';
import { OperationMapper } from '../operations/operation_mapper';
import { GraphExecutor } from './graph_executor';
import { ResourceManager } from './resource_manager';
export const TFHUB_SEARCH_PARAM = '?tfjs-format=file';
export const DEFAULT_MODEL_NAME = 'model.json';
/**
 * A `tf.GraphModel` is a directed, acyclic graph built from a
 * SavedModel GraphDef and allows inference execution.
 *
 * A `tf.GraphModel` can only be created by loading from a model converted from
 * a [TensorFlow SavedModel](https://www.tensorflow.org/guide/saved_model) using
 * the command line converter tool and loaded via `tf.loadGraphModel`.
 *
 * @doc {heading: 'Models', subheading: 'Classes'}
 */
export class GraphModel {
    /**
     * @param modelUrl url for the model, or an `io.IOHandler`.
     * @param weightManifestUrl url for the weight file generated by
     * scripts/convert.py script.
     * @param requestOption options for Request, which allows to send credentials
     * and custom headers.
     * @param onProgress Optional, progress callback function, fired periodically
     * before the load is completed.
     */
    constructor(modelUrl, loadOptions = {}, tfio = io) {
        this.modelUrl = modelUrl;
        this.loadOptions = loadOptions;
        this.version = 'n/a';
        this.io = tfio;
        if (loadOptions == null) {
            this.loadOptions = {};
        }
        this.resourceManager = new ResourceManager();
    }
    // Returns the version information for the tensorflow model GraphDef.
    get modelVersion() {
        return this.version;
    }
    get inputNodes() {
        return this.executor.inputNodes;
    }
    get outputNodes() {
        return this.executor.outputNodes;
    }
    get inputs() {
        return this.executor.inputs;
    }
    get outputs() {
        return this.executor.outputs;
    }
    get weights() {
        return this.executor.weightMap;
    }
    get metadata() {
        return this.artifacts.userDefinedMetadata;
    }
    get modelSignature() {
        return this.signature;
    }
    get modelStructuredOutputKeys() {
        return this.structuredOutputKeys;
    }
    findIOHandler() {
        const path = this.modelUrl;
        if (path.load != null) {
            // Path is an IO Handler.
            this.handler = path;
        }
        else if (this.loadOptions.requestInit != null) {
            this.handler = this.io.browserHTTPRequest(path, this.loadOptions);
        }
        else {
            const handlers = this.io.getLoadHandlers(path, this.loadOptions);
            if (handlers.length === 0) {
                // For backward compatibility: if no load handler can be found,
                // assume it is a relative http path.
                handlers.push(this.io.browserHTTPRequest(path, this.loadOptions));
            }
            else if (handlers.length > 1) {
                throw new Error(`Found more than one (${handlers.length}) load handlers for ` +
                    `URL '${[path]}'`);
            }
            this.handler = handlers[0];
        }
    }
    /**
     * Loads the model and weight files, construct the in memory weight map and
     * compile the inference graph.
     */
    load() {
        this.findIOHandler();
        if (this.handler.load == null) {
            throw new Error('Cannot proceed with model loading because the IOHandler provided ' +
                'does not have the `load` method implemented.');
        }
        const loadResult = this.handler.load();
        if (util.isPromise(loadResult)) {
            return loadResult.then(artifacts => this.loadSync(artifacts));
        }
        return this.loadSync(loadResult);
    }
    /**
     * Synchronously construct the in memory weight map and
     * compile the inference graph. Also initialize hashtable if any.
     *
     * @doc {heading: 'Models', subheading: 'Classes', ignoreCI: true}
     */
    loadSync(artifacts) {
        this.artifacts = artifacts;
        const graph = this.artifacts.modelTopology;
        let signature = this.artifacts.signature;
        if (this.artifacts.userDefinedMetadata != null) {
            const metadata = this.artifacts.userDefinedMetadata;
            if (metadata.signature != null) {
                signature = metadata.signature;
            }
            if (metadata.structuredOutputKeys != null) {
                this.structuredOutputKeys = metadata.structuredOutputKeys;
            }
        }
        this.signature = signature;
        this.version = `${graph.versions.producer}.${graph.versions.minConsumer}`;
        const weightMap = this.io.decodeWeights(this.artifacts.weightData, this.artifacts.weightSpecs);
        this.executor = new GraphExecutor(OperationMapper.Instance.transformGraph(graph, this.signature));
        this.executor.weightMap = this.convertTensorMapToTensorsMap(weightMap);
        // Attach a model-level resourceManager to each executor to share resources,
        // such as `HashTable`.
        this.executor.resourceManager = this.resourceManager;
        if (artifacts.modelInitializer != null &&
            artifacts.modelInitializer.node != null) {
            const initializer = OperationMapper.Instance.transformGraph(artifacts.modelInitializer);
            this.initializer = new GraphExecutor(initializer);
            this.initializer.weightMap = this.executor.weightMap;
            // Attach a model-level resourceManager to the initializer, the
            // hashTables created from when executing the initializer will be stored
            // in the resourceManager.
            this.initializer.resourceManager = this.resourceManager;
            this.initializer.executeAsync({}, []);
        }
        return true;
    }
    /**
     * Save the configuration and/or weights of the GraphModel.
     *
     * An `IOHandler` is an object that has a `save` method of the proper
     * signature defined. The `save` method manages the storing or
     * transmission of serialized data ("artifacts") that represent the
     * model's topology and weights onto or via a specific medium, such as
     * file downloads, local storage, IndexedDB in the web browser and HTTP
     * requests to a server. TensorFlow.js provides `IOHandler`
     * implementations for a number of frequently used saving mediums, such as
     * `tf.io.browserDownloads` and `tf.io.browserLocalStorage`. See `tf.io`
     * for more details.
     *
     * This method also allows you to refer to certain types of `IOHandler`s
     * as URL-like string shortcuts, such as 'localstorage://' and
     * 'indexeddb://'.
     *
     * Example 1: Save `model`'s topology and weights to browser [local
     * storage](https://developer.mozilla.org/en-US/docs/Web/API/Window/localStorage);
     * then load it back.
     *
     * ```js
     * const modelUrl =
     *    'https://storage.googleapis.com/tfjs-models/savedmodel/mobilenet_v2_1.0_224/model.json';
     * const model = await tf.loadGraphModel(modelUrl);
     * const zeros = tf.zeros([1, 224, 224, 3]);
     * model.predict(zeros).print();
     *
     * const saveResults = await model.save('localstorage://my-model-1');
     *
     * const loadedModel = await tf.loadGraphModel('localstorage://my-model-1');
     * console.log('Prediction from loaded model:');
     * model.predict(zeros).print();
     * ```
     *
     * @param handlerOrURL An instance of `IOHandler` or a URL-like,
     * scheme-based string shortcut for `IOHandler`.
     * @param config Options for saving the model.
     * @returns A `Promise` of `SaveResult`, which summarizes the result of
     * the saving, such as byte sizes of the saved artifacts for the model's
     *   topology and weight values.
     *
     * @doc {heading: 'Models', subheading: 'Classes', ignoreCI: true}
     */
    async save(handlerOrURL, config) {
        if (typeof handlerOrURL === 'string') {
            const handlers = this.io.getSaveHandlers(handlerOrURL);
            if (handlers.length === 0) {
                throw new Error(`Cannot find any save handlers for URL '${handlerOrURL}'`);
            }
            else if (handlers.length > 1) {
                throw new Error(`Found more than one (${handlers.length}) save handlers for ` +
                    `URL '${handlerOrURL}'`);
            }
            handlerOrURL = handlers[0];
        }
        if (handlerOrURL.save == null) {
            throw new Error('GraphModel.save() cannot proceed because the IOHandler ' +
                'provided does not have the `save` attribute defined.');
        }
        return handlerOrURL.save(this.artifacts);
    }
    /**
     * Execute the inference for the input tensors.
     *
     * @param input The input tensors, when there is single input for the model,
     * inputs param should be a `tf.Tensor`. For models with mutliple inputs,
     * inputs params should be in either `tf.Tensor`[] if the input order is
     * fixed, or otherwise NamedTensorMap format.
     *
     * For model with multiple inputs, we recommend you use NamedTensorMap as the
     * input type, if you use `tf.Tensor`[], the order of the array needs to
     * follow the
     * order of inputNodes array. @see {@link GraphModel.inputNodes}
     *
     * You can also feed any intermediate nodes using the NamedTensorMap as the
     * input type. For example, given the graph
     *    InputNode => Intermediate => OutputNode,
     * you can execute the subgraph Intermediate => OutputNode by calling
     *    model.execute('IntermediateNode' : tf.tensor(...));
     *
     * This is useful for models that uses tf.dynamic_rnn, where the intermediate
     * state needs to be fed manually.
     *
     * For batch inference execution, the tensors for each input need to be
     * concatenated together. For example with mobilenet, the required input shape
     * is [1, 244, 244, 3], which represents the [batch, height, width, channel].
     * If we are provide a batched data of 100 images, the input tensor should be
     * in the shape of [100, 244, 244, 3].
     *
     * @param config Prediction configuration for specifying the batch size.
     * Currently the batch size option is ignored for graph model.
     *
     * @returns Inference result tensors. If the model is converted and it
     * originally had structured_outputs in tensorflow, then a NamedTensorMap
     * will be returned matching the structured_outputs. If no structured_outputs
     * are present, the output will be single `tf.Tensor` if the model has single
     * output node, otherwise Tensor[].
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    predict(inputs, config) {
        const outputTensors = this.execute(inputs, this.outputNodes);
        if (this.structuredOutputKeys) {
            const outputTensorsArray = outputTensors instanceof Tensor ? [outputTensors] : outputTensors;
            const outputTensorMap = {};
            outputTensorsArray.forEach((outputTensor, i) => outputTensorMap[this.structuredOutputKeys[i]] =
                outputTensor);
            return outputTensorMap;
        }
        return outputTensors;
    }
    normalizeInputs(inputs) {
        if (!(inputs instanceof Tensor) && !Array.isArray(inputs)) {
            // The input is already a NamedTensorMap.
            return inputs;
        }
        inputs = Array.isArray(inputs) ? inputs : [inputs];
        if (inputs.length !== this.inputNodes.length) {
            throw new Error('Input tensor count mismatch,' +
                `the graph model has ${this.inputNodes.length} placeholders, ` +
                `while there are ${inputs.length} input tensors.`);
        }
        return this.inputNodes.reduce((map, inputName, i) => {
            map[inputName] = inputs[i];
            return map;
        }, {});
    }
    normalizeOutputs(outputs) {
        outputs = outputs || this.outputNodes;
        return !Array.isArray(outputs) ? [outputs] : outputs;
    }
    /**
     * Executes inference for the model for given input tensors.
     * @param inputs tensor, tensor array or tensor map of the inputs for the
     * model, keyed by the input node names.
     * @param outputs output node name from the Tensorflow model, if no
     * outputs are specified, the default outputs of the model would be used.
     * You can inspect intermediate nodes of the model by adding them to the
     * outputs array.
     *
     * @returns A single tensor if provided with a single output or no outputs
     * are provided and there is only one default output, otherwise return a
     * tensor array. The order of the tensor array is the same as the outputs
     * if provided, otherwise the order of outputNodes attribute of the model.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    execute(inputs, outputs) {
        inputs = this.normalizeInputs(inputs);
        outputs = this.normalizeOutputs(outputs);
        const result = this.executor.execute(inputs, outputs);
        return result.length > 1 ? result : result[0];
    }
    /**
     * Executes inference for the model for given input tensors in async
     * fashion, use this method when your model contains control flow ops.
     * @param inputs tensor, tensor array or tensor map of the inputs for the
     * model, keyed by the input node names.
     * @param outputs output node name from the Tensorflow model, if no outputs
     * are specified, the default outputs of the model would be used. You can
     * inspect intermediate nodes of the model by adding them to the outputs
     * array.
     *
     * @returns A Promise of single tensor if provided with a single output or
     * no outputs are provided and there is only one default output, otherwise
     * return a tensor map.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    async executeAsync(inputs, outputs) {
        inputs = this.normalizeInputs(inputs);
        outputs = this.normalizeOutputs(outputs);
        const result = await this.executor.executeAsync(inputs, outputs);
        return result.length > 1 ? result : result[0];
    }
    /**
     * Get intermediate tensors for model debugging mode (flag
     * KEEP_INTERMEDIATE_TENSORS is true).
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    getIntermediateTensors() {
        return this.executor.getIntermediateTensors();
    }
    /**
     * Dispose intermediate tensors for model debugging mode (flag
     * KEEP_INTERMEDIATE_TENSORS is true).
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    disposeIntermediateTensors() {
        this.executor.disposeIntermediateTensors();
    }
    convertTensorMapToTensorsMap(map) {
        return Object.keys(map).reduce((newMap, key) => {
            newMap[key] = [map[key]];
            return newMap;
        }, {});
    }
    /**
     * Releases the memory used by the weight tensors and resourceManager.
     *
     * @doc {heading: 'Models', subheading: 'Classes'}
     */
    dispose() {
        this.executor.dispose();
        if (this.initializer) {
            this.initializer.dispose();
        }
        this.resourceManager.dispose();
    }
}
/**
 * Load a graph model given a URL to the model definition.
 *
 * Example of loading MobileNetV2 from a URL and making a prediction with a
 * zeros input:
 *
 * ```js
 * const modelUrl =
 *    'https://storage.googleapis.com/tfjs-models/savedmodel/mobilenet_v2_1.0_224/model.json';
 * const model = await tf.loadGraphModel(modelUrl);
 * const zeros = tf.zeros([1, 224, 224, 3]);
 * model.predict(zeros).print();
 * ```
 *
 * Example of loading MobileNetV2 from a TF Hub URL and making a prediction
 * with a zeros input:
 *
 * ```js
 * const modelUrl =
 *    'https://tfhub.dev/google/imagenet/mobilenet_v2_140_224/classification/2';
 * const model = await tf.loadGraphModel(modelUrl, {fromTFHub: true});
 * const zeros = tf.zeros([1, 224, 224, 3]);
 * model.predict(zeros).print();
 * ```
 * @param modelUrl The url or an `io.IOHandler` that loads the model.
 * @param options Options for the HTTP request, which allows to send
 *     credentials
 *    and custom headers.
 *
 * @doc {heading: 'Models', subheading: 'Loading'}
 */
export async function loadGraphModel(modelUrl, options = {}, tfio = io) {
    if (modelUrl == null) {
        throw new Error('modelUrl in loadGraphModel() cannot be null. Please provide a url ' +
            'or an IOHandler that loads the model');
    }
    if (options == null) {
        options = {};
    }
    if (options.fromTFHub && typeof modelUrl === 'string') {
        modelUrl = getTFHubUrl(modelUrl);
    }
    const model = new GraphModel(modelUrl, options, tfio);
    await model.load();
    return model;
}
/**
 * Load a graph model given a synchronous IO handler with a 'load' method.
 *
 * @param modelSource The `io.IOHandlerSync` that loads the model.
 *
 * @doc {heading: 'Models', subheading: 'Loading'}
 */
export function loadGraphModelSync(modelSource) {
    if (modelSource == null) {
        throw new Error('modelUrl in loadGraphModelSync() cannot be null. Please provide a ' +
            'url or an IOHandler that loads the model');
    }
    if (!modelSource.load) {
        throw new Error(`modelUrl IO Handler ${modelSource} has no load function`);
    }
    const model = new GraphModel(modelSource);
    model.load();
    return model;
}
function getTFHubUrl(modelUrl) {
    if (!modelUrl.endsWith('/')) {
        modelUrl = (modelUrl) + '/';
    }
    return `${modelUrl}${DEFAULT_MODEL_NAME}${TFHUB_SEARCH_PARAM}`;
}
//# sourceMappingURL=data:application/json;base64,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