/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from '@tensorflow/tfjs-core';
import { getInternalFormatForFloat16MatrixTexture, getInternalFormatForFloat16PackedMatrixTexture, getInternalFormatForFloat32MatrixTexture, getInternalFormatForPackedMatrixTexture, getInternalFormatForUnsignedBytesMatrixTexture } from './gpgpu_util';
import { getPackedMatrixTextureShapeWidthHeight, getUnpackedMatrixTextureShapeWidthHeight, PhysicalTextureType, TextureUsage } from './tex_util';
export class TextureManager {
    constructor(gpgpu) {
        this.gpgpu = gpgpu;
        this.numUsedTextures = 0;
        this.numFreeTextures = 0;
        this._numBytesAllocated = 0;
        this._numBytesFree = 0; // How many bytes that have been allocated
        // are available for reuse.
        this.freeTextures = {};
        this.logEnabled = false;
        this.usedTextures = {};
    }
    acquireTexture(shapeRC, usage, isPacked) {
        const physicalTexType = getPhysicalFromLogicalTextureType(usage, isPacked);
        const shapeKey = getKeyFromTextureShape(shapeRC, physicalTexType, isPacked);
        if (!(shapeKey in this.freeTextures)) {
            this.freeTextures[shapeKey] = [];
        }
        if (!(shapeKey in this.usedTextures)) {
            this.usedTextures[shapeKey] = [];
        }
        const texBytes = computeBytes(shapeRC, physicalTexType, this.gpgpu.gl, this.gpgpu.textureConfig, isPacked);
        if (this.freeTextures[shapeKey].length > 0) {
            this.numFreeTextures--;
            this.numUsedTextures++;
            this._numBytesFree -= texBytes;
            this.log();
            const newTexture = this.freeTextures[shapeKey].shift();
            this.usedTextures[shapeKey].push(newTexture);
            return newTexture;
        }
        let newTexture;
        if (physicalTexType === PhysicalTextureType.PACKED_2X2_FLOAT32) {
            newTexture = this.gpgpu.createPackedMatrixTexture(shapeRC[0], shapeRC[1]);
        }
        else if (physicalTexType === PhysicalTextureType.PACKED_2X2_FLOAT16) {
            newTexture =
                this.gpgpu.createFloat16PackedMatrixTexture(shapeRC[0], shapeRC[1]);
        }
        else if (physicalTexType === PhysicalTextureType.UNPACKED_FLOAT32) {
            newTexture =
                this.gpgpu.createFloat32MatrixTexture(shapeRC[0], shapeRC[1]);
        }
        else if (physicalTexType === PhysicalTextureType.UNPACKED_FLOAT16) {
            newTexture =
                this.gpgpu.createFloat16MatrixTexture(shapeRC[0], shapeRC[1]);
        }
        else if (physicalTexType === PhysicalTextureType.PACKED_4X1_UNSIGNED_BYTE) {
            newTexture =
                this.gpgpu.createUnsignedBytesMatrixTexture(shapeRC[0], shapeRC[1]);
        }
        this.usedTextures[shapeKey].push(newTexture);
        this.numUsedTextures++;
        this._numBytesAllocated += texBytes;
        this.log();
        return newTexture;
    }
    releaseTexture(texture, shape, logicalTexType, isPacked) {
        if (this.freeTextures == null) {
            // Already disposed.
            return;
        }
        const physicalTexType = getPhysicalFromLogicalTextureType(logicalTexType, isPacked);
        const shapeKey = getKeyFromTextureShape(shape, physicalTexType, isPacked);
        if (!(shapeKey in this.freeTextures)) {
            this.freeTextures[shapeKey] = [];
        }
        const texBytes = computeBytes(shape, physicalTexType, this.gpgpu.gl, this.gpgpu.textureConfig, isPacked);
        const deleteTexThreshold = env().get('WEBGL_DELETE_TEXTURE_THRESHOLD');
        if (deleteTexThreshold !== -1 &&
            this._numBytesAllocated > deleteTexThreshold) {
            this.gpgpu.deleteMatrixTexture(texture.texture);
            this._numBytesAllocated -= texBytes;
        }
        else {
            this.freeTextures[shapeKey].push(texture);
            this.numFreeTextures++;
            this._numBytesFree += texBytes;
        }
        this.numUsedTextures--;
        const texList = this.usedTextures[shapeKey];
        const texIndex = texList.indexOf(texture);
        if (texIndex < 0) {
            throw new Error('Cannot release a texture that was never provided by this ' +
                'texture manager');
        }
        texList.splice(texIndex, 1);
        this.log();
    }
    log() {
        if (!this.logEnabled) {
            return;
        }
        const total = this.numFreeTextures + this.numUsedTextures;
        console.log('Free/Used', `${this.numFreeTextures} / ${this.numUsedTextures}`, `(${total})`);
        const freeRatio = this._numBytesFree / this._numBytesAllocated;
        console.log(`Bytes allocated: ${this._numBytesAllocated}`);
        console.log(`Bytes unused: ${this._numBytesFree} (${Math.round(100 * freeRatio)}%)`);
    }
    get numBytesAllocated() {
        return this._numBytesAllocated;
    }
    get numBytesFree() {
        return this._numBytesFree;
    }
    getNumUsedTextures() {
        return this.numUsedTextures;
    }
    getNumFreeTextures() {
        return this.numFreeTextures;
    }
    dispose() {
        if (this.freeTextures == null) {
            // Already disposed.
            return;
        }
        for (const texShape in this.freeTextures) {
            this.freeTextures[texShape].forEach(tex => {
                this.gpgpu.deleteMatrixTexture(tex.texture);
            });
        }
        for (const texShape in this.usedTextures) {
            this.usedTextures[texShape].forEach(tex => {
                this.gpgpu.deleteMatrixTexture(tex.texture);
            });
        }
        this.freeTextures = null;
        this.usedTextures = null;
        this.numUsedTextures = 0;
        this.numFreeTextures = 0;
        this._numBytesAllocated = 0;
        this._numBytesFree = 0;
    }
}
function numBytesForInternalFormat(gl, internalFormat) {
    // tslint:disable-next-line:no-any
    const glany = gl;
    if (internalFormat === glany.R32F) {
        return 4;
    }
    else if (internalFormat === glany.R16F) {
        return 2;
    }
    else if (internalFormat === glany.RGBA32F) {
        return 16;
    }
    else if (internalFormat === gl.RGBA) {
        return 16;
    }
    else if (internalFormat === glany.RGBA16F) {
        return 8;
    }
    else if (internalFormat === glany.RGBA8) {
        return 4;
    }
    throw new Error(`Unknown internal format ${internalFormat}`);
}
export function computeBytes(shape, physicalTexType, gl, textureConfig, isPacked) {
    // It is not possible to infer packed status from the texture type because
    // depending on the textureConfig, different  texture types may resolve to the
    // same internal format (e.g. in WebGL1, the internal format for
    // UNPACKED_FLOAT16 textures is gl.RGBA). Therefore we pass in `isPacked`
    // explicitly.
    const internalFormat = internalFormatForPhysicalTexType(physicalTexType, textureConfig);
    let numElements;
    if (isPacked) {
        const [packedWidth, packedHeight] = getPackedMatrixTextureShapeWidthHeight(shape[0], shape[1]);
        numElements = packedWidth * packedHeight;
    }
    else {
        const [width, height] = getUnpackedMatrixTextureShapeWidthHeight(shape[0], shape[1]);
        numElements = width * height;
    }
    const bytesPerElement = numBytesForInternalFormat(gl, internalFormat);
    return numElements * bytesPerElement;
}
function internalFormatForPhysicalTexType(physicalTexType, textureConfig) {
    switch (physicalTexType) {
        case PhysicalTextureType.PACKED_2X2_FLOAT32:
            return getInternalFormatForPackedMatrixTexture(textureConfig);
        case PhysicalTextureType.PACKED_2X2_FLOAT16:
            return getInternalFormatForFloat16PackedMatrixTexture(textureConfig);
        case PhysicalTextureType.UNPACKED_FLOAT32:
            return getInternalFormatForFloat32MatrixTexture(textureConfig);
        case PhysicalTextureType.UNPACKED_FLOAT16:
            return getInternalFormatForFloat16MatrixTexture(textureConfig);
        case PhysicalTextureType.PACKED_4X1_UNSIGNED_BYTE:
            return getInternalFormatForUnsignedBytesMatrixTexture(textureConfig);
        default:
            throw new Error(`Unknown physical texture type ${physicalTexType}`);
    }
}
function getPhysicalTextureForRendering(isPacked) {
    if (env().getBool('WEBGL_RENDER_FLOAT32_ENABLED')) {
        if (isPacked) {
            return PhysicalTextureType.PACKED_2X2_FLOAT32;
        }
        return PhysicalTextureType.UNPACKED_FLOAT32;
    }
    if (isPacked) {
        return PhysicalTextureType.PACKED_2X2_FLOAT16;
    }
    return PhysicalTextureType.UNPACKED_FLOAT16;
}
function getPhysicalFromLogicalTextureType(logicalTexType, isPacked) {
    if (logicalTexType === TextureUsage.UPLOAD) {
        return PhysicalTextureType.PACKED_2X2_FLOAT32;
    }
    else if (logicalTexType === TextureUsage.RENDER || logicalTexType == null) {
        return getPhysicalTextureForRendering(isPacked);
    }
    else if (logicalTexType === TextureUsage.DOWNLOAD ||
        logicalTexType === TextureUsage.PIXELS) {
        return PhysicalTextureType.PACKED_4X1_UNSIGNED_BYTE;
    }
    throw new Error(`Unknown logical texture type ${logicalTexType}`);
}
function getKeyFromTextureShape(shapeRowsCol, physicalTexType, isPacked) {
    return `${shapeRowsCol[0]}_${shapeRowsCol[1]}_${physicalTexType}_${isPacked}`;
}
//# sourceMappingURL=data:application/json;base64,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