/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from '@tensorflow/tfjs-core';
import { FromPixels } from '@tensorflow/tfjs-core';
import { TextureUsage } from '../tex_util';
import { FromPixelsProgram } from './FromPixels_utils/from_pixels_gpu';
import { FromPixelsPackedProgram } from './FromPixels_utils/from_pixels_packed_gpu';
export const fromPixelsConfig = {
    kernelName: FromPixels,
    backendName: 'webgl',
    kernelFunc: fromPixels,
};
let fromPixels2DContext;
function fromPixels(args) {
    const { inputs, backend, attrs } = args;
    let { pixels } = inputs;
    const { numChannels } = attrs;
    const isVideo = typeof (HTMLVideoElement) !== 'undefined' &&
        pixels instanceof HTMLVideoElement;
    const isImage = typeof (HTMLImageElement) !== 'undefined' &&
        pixels instanceof HTMLImageElement;
    const [width, height] = isVideo ?
        [
            pixels.videoWidth,
            pixels.videoHeight
        ] :
        [pixels.width, pixels.height];
    const texShape = [height, width];
    const outShape = [height, width, numChannels];
    if (isImage || isVideo) {
        if (fromPixels2DContext == null) {
            fromPixels2DContext = document.createElement('canvas').getContext('2d');
        }
        fromPixels2DContext.canvas.width = width;
        fromPixels2DContext.canvas.height = height;
        fromPixels2DContext.drawImage(pixels, 0, 0, width, height);
        pixels = fromPixels2DContext.canvas;
    }
    const tempPixelHandle = backend.makeTensorInfo(texShape, 'int32');
    // This is a byte texture with pixels.
    backend.texData.get(tempPixelHandle.dataId).usage = TextureUsage.PIXELS;
    backend.gpgpu.uploadPixelDataToTexture(backend.getTexture(tempPixelHandle.dataId), pixels);
    const program = env().getBool('WEBGL_PACK') ?
        new FromPixelsPackedProgram(outShape) :
        new FromPixelsProgram(outShape);
    const res = backend.runWebGLProgram(program, [tempPixelHandle], 'int32');
    backend.disposeData(tempPixelHandle.dataId);
    return res;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRnJvbVBpeGVscy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uL3RmanMtYmFja2VuZC13ZWJnbC9zcmMva2VybmVscy9Gcm9tUGl4ZWxzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUVILE9BQU8sRUFBQyxHQUFHLEVBQXVDLE1BQU0sdUJBQXVCLENBQUM7QUFDaEYsT0FBTyxFQUFDLFVBQVUsRUFBb0MsTUFBTSx1QkFBdUIsQ0FBQztBQUdwRixPQUFPLEVBQUMsWUFBWSxFQUFDLE1BQU0sYUFBYSxDQUFDO0FBRXpDLE9BQU8sRUFBQyxpQkFBaUIsRUFBQyxNQUFNLG9DQUFvQyxDQUFDO0FBQ3JFLE9BQU8sRUFBQyx1QkFBdUIsRUFBQyxNQUFNLDJDQUEyQyxDQUFDO0FBRWxGLE1BQU0sQ0FBQyxNQUFNLGdCQUFnQixHQUFpQjtJQUM1QyxVQUFVLEVBQUUsVUFBVTtJQUN0QixXQUFXLEVBQUUsT0FBTztJQUNwQixVQUFVLEVBQUUsVUFBOEI7Q0FDM0MsQ0FBQztBQUVGLElBQUksbUJBQTZDLENBQUM7QUFFbEQsU0FBUyxVQUFVLENBQUMsSUFJbkI7SUFDQyxNQUFNLEVBQUMsTUFBTSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUMsR0FBRyxJQUFJLENBQUM7SUFDdEMsSUFBSSxFQUFDLE1BQU0sRUFBQyxHQUFHLE1BQU0sQ0FBQztJQUN0QixNQUFNLEVBQUMsV0FBVyxFQUFDLEdBQUcsS0FBSyxDQUFDO0lBRTVCLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLFdBQVc7UUFDckQsTUFBTSxZQUFZLGdCQUFnQixDQUFDO0lBQ3ZDLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLFdBQVc7UUFDckQsTUFBTSxZQUFZLGdCQUFnQixDQUFDO0lBQ3ZDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEdBQUcsT0FBTyxDQUFDLENBQUM7UUFDN0I7WUFDRyxNQUEyQixDQUFDLFVBQVU7WUFDdEMsTUFBMkIsQ0FBQyxXQUFXO1NBQ3pDLENBQUMsQ0FBQztRQUNILENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFbEMsTUFBTSxRQUFRLEdBQXFCLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ25ELE1BQU0sUUFBUSxHQUFHLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztJQUU5QyxJQUFJLE9BQU8sSUFBSSxPQUFPLEVBQUU7UUFDdEIsSUFBSSxtQkFBbUIsSUFBSSxJQUFJLEVBQUU7WUFDL0IsbUJBQW1CLEdBQUcsUUFBUSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDekU7UUFFRCxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUN6QyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztRQUMzQyxtQkFBbUIsQ0FBQyxTQUFTLENBQ3pCLE1BQTJELEVBQzNELENBQUMsRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxtQkFBbUIsQ0FBQyxNQUFNLENBQUM7S0FDckM7SUFFRCxNQUFNLGVBQWUsR0FBRyxPQUFPLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRSxzQ0FBc0M7SUFDdEMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssR0FBRyxZQUFZLENBQUMsTUFBTSxDQUFDO0lBQ3hFLE9BQU8sQ0FBQyxLQUFLLENBQUMsd0JBQXdCLENBQ2xDLE9BQU8sQ0FBQyxVQUFVLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQW1CLENBQUMsQ0FBQztJQUNyRSxNQUFNLE9BQU8sR0FBRyxHQUFHLEVBQUUsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUN6QyxJQUFJLHVCQUF1QixDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDdkMsSUFBSSxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNwQyxNQUFNLEdBQUcsR0FBRyxPQUFPLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3pFLE9BQU8sQ0FBQyxXQUFXLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzVDLE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCAyMDE5IEdvb2dsZSBMTEMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpO1xuICogeW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuICogWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG4gKlxuICogaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuICogZGlzdHJpYnV0ZWQgdW5kZXIgdGhlIExpY2Vuc2UgaXMgZGlzdHJpYnV0ZWQgb24gYW4gXCJBUyBJU1wiIEJBU0lTLFxuICogV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gKiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gKiBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqID09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09XG4gKi9cblxuaW1wb3J0IHtlbnYsIEtlcm5lbENvbmZpZywgS2VybmVsRnVuYywgVGVuc29ySW5mb30gZnJvbSAnQHRlbnNvcmZsb3cvdGZqcy1jb3JlJztcbmltcG9ydCB7RnJvbVBpeGVscywgRnJvbVBpeGVsc0F0dHJzLCBGcm9tUGl4ZWxzSW5wdXRzfSBmcm9tICdAdGVuc29yZmxvdy90ZmpzLWNvcmUnO1xuXG5pbXBvcnQge01hdGhCYWNrZW5kV2ViR0x9IGZyb20gJy4uL2JhY2tlbmRfd2ViZ2wnO1xuaW1wb3J0IHtUZXh0dXJlVXNhZ2V9IGZyb20gJy4uL3RleF91dGlsJztcblxuaW1wb3J0IHtGcm9tUGl4ZWxzUHJvZ3JhbX0gZnJvbSAnLi9Gcm9tUGl4ZWxzX3V0aWxzL2Zyb21fcGl4ZWxzX2dwdSc7XG5pbXBvcnQge0Zyb21QaXhlbHNQYWNrZWRQcm9ncmFtfSBmcm9tICcuL0Zyb21QaXhlbHNfdXRpbHMvZnJvbV9waXhlbHNfcGFja2VkX2dwdSc7XG5cbmV4cG9ydCBjb25zdCBmcm9tUGl4ZWxzQ29uZmlnOiBLZXJuZWxDb25maWcgPSB7XG4gIGtlcm5lbE5hbWU6IEZyb21QaXhlbHMsXG4gIGJhY2tlbmROYW1lOiAnd2ViZ2wnLFxuICBrZXJuZWxGdW5jOiBmcm9tUGl4ZWxzIGFzIHt9IGFzIEtlcm5lbEZ1bmMsXG59O1xuXG5sZXQgZnJvbVBpeGVsczJEQ29udGV4dDogQ2FudmFzUmVuZGVyaW5nQ29udGV4dDJEO1xuXG5mdW5jdGlvbiBmcm9tUGl4ZWxzKGFyZ3M6IHtcbiAgaW5wdXRzOiBGcm9tUGl4ZWxzSW5wdXRzLFxuICBiYWNrZW5kOiBNYXRoQmFja2VuZFdlYkdMLFxuICBhdHRyczogRnJvbVBpeGVsc0F0dHJzXG59KTogVGVuc29ySW5mbyB7XG4gIGNvbnN0IHtpbnB1dHMsIGJhY2tlbmQsIGF0dHJzfSA9IGFyZ3M7XG4gIGxldCB7cGl4ZWxzfSA9IGlucHV0cztcbiAgY29uc3Qge251bUNoYW5uZWxzfSA9IGF0dHJzO1xuXG4gIGNvbnN0IGlzVmlkZW8gPSB0eXBlb2YgKEhUTUxWaWRlb0VsZW1lbnQpICE9PSAndW5kZWZpbmVkJyAmJlxuICAgICAgcGl4ZWxzIGluc3RhbmNlb2YgSFRNTFZpZGVvRWxlbWVudDtcbiAgY29uc3QgaXNJbWFnZSA9IHR5cGVvZiAoSFRNTEltYWdlRWxlbWVudCkgIT09ICd1bmRlZmluZWQnICYmXG4gICAgICBwaXhlbHMgaW5zdGFuY2VvZiBIVE1MSW1hZ2VFbGVtZW50O1xuICBjb25zdCBbd2lkdGgsIGhlaWdodF0gPSBpc1ZpZGVvID9cbiAgICAgIFtcbiAgICAgICAgKHBpeGVscyBhcyBIVE1MVmlkZW9FbGVtZW50KS52aWRlb1dpZHRoLFxuICAgICAgICAocGl4ZWxzIGFzIEhUTUxWaWRlb0VsZW1lbnQpLnZpZGVvSGVpZ2h0XG4gICAgICBdIDpcbiAgICAgIFtwaXhlbHMud2lkdGgsIHBpeGVscy5oZWlnaHRdO1xuXG4gIGNvbnN0IHRleFNoYXBlOiBbbnVtYmVyLCBudW1iZXJdID0gW2hlaWdodCwgd2lkdGhdO1xuICBjb25zdCBvdXRTaGFwZSA9IFtoZWlnaHQsIHdpZHRoLCBudW1DaGFubmVsc107XG5cbiAgaWYgKGlzSW1hZ2UgfHwgaXNWaWRlbykge1xuICAgIGlmIChmcm9tUGl4ZWxzMkRDb250ZXh0ID09IG51bGwpIHtcbiAgICAgIGZyb21QaXhlbHMyRENvbnRleHQgPSBkb2N1bWVudC5jcmVhdGVFbGVtZW50KCdjYW52YXMnKS5nZXRDb250ZXh0KCcyZCcpO1xuICAgIH1cblxuICAgIGZyb21QaXhlbHMyRENvbnRleHQuY2FudmFzLndpZHRoID0gd2lkdGg7XG4gICAgZnJvbVBpeGVsczJEQ29udGV4dC5jYW52YXMuaGVpZ2h0ID0gaGVpZ2h0O1xuICAgIGZyb21QaXhlbHMyRENvbnRleHQuZHJhd0ltYWdlKFxuICAgICAgICBwaXhlbHMgYXMgSFRNTFZpZGVvRWxlbWVudCB8IEhUTUxJbWFnZUVsZW1lbnQgfCBJbWFnZUJpdG1hcCxcbiAgICAgICAgMCwgMCwgd2lkdGgsIGhlaWdodCk7XG4gICAgcGl4ZWxzID0gZnJvbVBpeGVsczJEQ29udGV4dC5jYW52YXM7XG4gIH1cblxuICBjb25zdCB0ZW1wUGl4ZWxIYW5kbGUgPSBiYWNrZW5kLm1ha2VUZW5zb3JJbmZvKHRleFNoYXBlLCAnaW50MzInKTtcbiAgLy8gVGhpcyBpcyBhIGJ5dGUgdGV4dHVyZSB3aXRoIHBpeGVscy5cbiAgYmFja2VuZC50ZXhEYXRhLmdldCh0ZW1wUGl4ZWxIYW5kbGUuZGF0YUlkKS51c2FnZSA9IFRleHR1cmVVc2FnZS5QSVhFTFM7XG4gIGJhY2tlbmQuZ3BncHUudXBsb2FkUGl4ZWxEYXRhVG9UZXh0dXJlKFxuICAgICAgYmFja2VuZC5nZXRUZXh0dXJlKHRlbXBQaXhlbEhhbmRsZS5kYXRhSWQpLCBwaXhlbHMgYXMgSW1hZ2VEYXRhKTtcbiAgY29uc3QgcHJvZ3JhbSA9IGVudigpLmdldEJvb2woJ1dFQkdMX1BBQ0snKSA/XG4gICAgICBuZXcgRnJvbVBpeGVsc1BhY2tlZFByb2dyYW0ob3V0U2hhcGUpIDpcbiAgICAgIG5ldyBGcm9tUGl4ZWxzUHJvZ3JhbShvdXRTaGFwZSk7XG4gIGNvbnN0IHJlcyA9IGJhY2tlbmQucnVuV2ViR0xQcm9ncmFtKHByb2dyYW0sIFt0ZW1wUGl4ZWxIYW5kbGVdLCAnaW50MzInKTtcbiAgYmFja2VuZC5kaXNwb3NlRGF0YSh0ZW1wUGl4ZWxIYW5kbGUuZGF0YUlkKTtcbiAgcmV0dXJuIHJlcztcbn1cbiJdfQ==