/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { device_util, env } from '@tensorflow/tfjs-core';
import { getMaxTexturesInShader, getWebGLDisjointQueryTimerVersion, getWebGLMaxTextureSize, isCapableOfRenderingToFloatTexture, isDownloadFloatTextureEnabled, isWebGLFenceEnabled, isWebGLVersionEnabled } from './webgl_util';
const ENV = env();
/**
 * This file contains WebGL-specific flag registrations.
 */
/**
 * True if WebGL is supported.
 */
ENV.registerFlag('HAS_WEBGL', () => ENV.getNumber('WEBGL_VERSION') > 0);
/** 0: No WebGL, 1: WebGL 1.0, 2: WebGL 2.0. */
ENV.registerFlag('WEBGL_VERSION', () => {
    if (isWebGLVersionEnabled(2)) {
        return 2;
    }
    else if (isWebGLVersionEnabled(1)) {
        return 1;
    }
    return 0;
});
/** Whether to check for numerical representation problems. */
ENV.registerFlag('WEBGL_CHECK_NUMERICAL_PROBLEMS', () => false);
ENV.registerFlag('WEBGL_BUFFER_SUPPORTED', () => ENV.get('WEBGL_VERSION') === 2);
/** Whether the WebGL backend will sometimes forward ops to the CPU. */
ENV.registerFlag('WEBGL_CPU_FORWARD', () => true);
/** Whether the WebGL backend will always use f16 textures for rendering. */
ENV.registerFlag('WEBGL_FORCE_F16_TEXTURES', () => false);
/** Whether to turn all packing related flags on. */
ENV.registerFlag('WEBGL_PACK', () => ENV.getBool('HAS_WEBGL'));
/** Whether we will pack the batchnormalization op. */
ENV.registerFlag('WEBGL_PACK_NORMALIZATION', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack the clip op. */
ENV.registerFlag('WEBGL_PACK_CLIP', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack the depthwise conv op. */
ENV.registerFlag('WEBGL_PACK_DEPTHWISECONV', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack binary ops. */
ENV.registerFlag('WEBGL_PACK_BINARY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack unary ops. */
ENV.registerFlag('WEBGL_PACK_UNARY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack array ops. */
ENV.registerFlag('WEBGL_PACK_ARRAY_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack image ops. */
ENV.registerFlag('WEBGL_PACK_IMAGE_OPERATIONS', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will pack reduce ops. */
ENV.registerFlag('WEBGL_PACK_REDUCE', () => ENV.getBool('WEBGL_PACK'));
/** Whether packed WebGL kernels lazily unpack their outputs. */
ENV.registerFlag('WEBGL_LAZILY_UNPACK', () => ENV.getBool('WEBGL_PACK'));
/** Whether we will use the im2col algorithm to speed up convolutions. */
ENV.registerFlag('WEBGL_CONV_IM2COL', () => ENV.getBool('WEBGL_PACK'));
/** The maximum texture dimension. */
ENV.registerFlag('WEBGL_MAX_TEXTURE_SIZE', () => getWebGLMaxTextureSize(ENV.getNumber('WEBGL_VERSION')));
/** The maximum texture dimension. */
ENV.registerFlag('WEBGL_MAX_TEXTURES_IN_SHADER', () => getMaxTexturesInShader(ENV.getNumber('WEBGL_VERSION')));
/**
 * The disjoint_query_timer extension version.
 * 0: disabled, 1: EXT_disjoint_timer_query, 2:
 * EXT_disjoint_timer_query_webgl2.
 * In Firefox with WebGL 2.0,
 * EXT_disjoint_timer_query_webgl2 is not available, so we must use the
 * WebGL 1.0 extension.
 */
ENV.registerFlag('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION', () => {
    const webGLVersion = ENV.getNumber('WEBGL_VERSION');
    if (webGLVersion === 0) {
        return 0;
    }
    return getWebGLDisjointQueryTimerVersion(webGLVersion);
});
/**
 * Whether the timer object from the disjoint_query_timer extension gives
 * timing information that is reliable.
 */
ENV.registerFlag('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_RELIABLE', () => ENV.getNumber('WEBGL_DISJOINT_QUERY_TIMER_EXTENSION_VERSION') > 0 &&
    !device_util.isMobile());
/**
 * Whether the device is physically capable of rendering to float32 textures.
 */
ENV.registerFlag('WEBGL_RENDER_FLOAT32_CAPABLE', () => isCapableOfRenderingToFloatTexture(ENV.getNumber('WEBGL_VERSION')));
/**
 * Whether rendering to float32 textures is enabled. If disabled, renders to
 * float16 textures.
 */
ENV.registerFlag('WEBGL_RENDER_FLOAT32_ENABLED', () => {
    return ENV.getBool('WEBGL_FORCE_F16_TEXTURES') ?
        false :
        ENV.getBool('WEBGL_RENDER_FLOAT32_CAPABLE');
});
/**
 * Whether downloading float textures is enabled (16 or 32 bit). If disabled,
 * uses IEEE 754 encoding of the float32 values to 4 uint8 when downloading.
 */
ENV.registerFlag('WEBGL_DOWNLOAD_FLOAT_ENABLED', () => isDownloadFloatTextureEnabled(ENV.getNumber('WEBGL_VERSION')));
/** Whether the fence API is available. */
ENV.registerFlag('WEBGL_FENCE_API_ENABLED', () => isWebGLFenceEnabled(ENV.getNumber('WEBGL_VERSION')));
/**
 * Tensors with size <= than this will be uploaded as uniforms, not textures.
 */
ENV.registerFlag('WEBGL_SIZE_UPLOAD_UNIFORM', () => {
    // Use uniform uploads only when 32bit floats are supported. In
    // 16bit
    // environments there are problems with comparing a 16bit texture value
    // with a 32bit uniform value.
    const useUniforms = ENV.getBool('WEBGL_RENDER_FLOAT32_ENABLED');
    return useUniforms ? 4 : 0;
});
/**
 * If the total number of bytes allocated on the GPU is greater than this
 * number, we will aggressively delete textures upon disposal with
 * gl.deleteMatrixTexture, rather than making them available for reuse.
 *
 * Default value -1 indicates that we will never aggressively delete textures.
 */
ENV.registerFlag('WEBGL_DELETE_TEXTURE_THRESHOLD', () => {
    return -1;
}, threshold => {
    if (threshold < 0 && threshold !== -1) {
        throw new Error(`WEBGL_DELETE_TEXTURE_THRESHOLD must be -1 (indicating never ` +
            `delete) or at least 0, but got ${threshold}.`);
    }
});
/**
 * Trigger a manual GL command flush if the threshold of time has passed since
 * previous Kernel execution. This can be useful for Andorid device where GL
 * command flush are delayed un til the end of javascript task. This value is
 * measured in millisecond. Typically you want to set this value to close to 1.
 *
 * Default value 1 for mobile chrome, and -1 for rest cases. -1 indicates that
 * we will not enforce manual flush and depend on system default flush schedule.
 */
ENV.registerFlag('WEBGL_FLUSH_THRESHOLD', () => {
    return device_util.isMobile() ? 1 : -1;
}, threshold => {
    if (threshold < 0 && threshold !== -1) {
        throw new Error(`WEBGL_FLUSH_THRESHOLD must be -1 (indicating never ` +
            `manual flush) or at least 0, but got ${threshold}.`);
    }
});
/**
 * Threshold for input tensor size that determines whether WebGL backend will
 * delegate computation to CPU.
 *
 * Default value is 128.
 */
ENV.registerFlag('CPU_HANDOFF_SIZE_THRESHOLD', () => 128);
/** Whether we will use shapes uniforms. */
ENV.registerFlag('WEBGL_USE_SHAPES_UNIFORMS', () => false);
/**
 * Threshold for last dimension of input tensor that determines whether
 * WebGL backend for the Top K op will delegate computation to CPU. If input
 * is smaller than threshold then CPU will be used
 *
 * Default value is 100000.
 */
ENV.registerFlag('TOPK_LAST_DIM_CPU_HANDOFF_SIZE_THRESHOLD', () => 100000);
/**
 * Threshold for K that determines whether
 * WebGL backend for the Top K op will delegate computation to CPU. If k
 * is larger than threshold then CPU will be used
 *
 * Default value is 128.
 */
ENV.registerFlag('TOPK_K_CPU_HANDOFF_THRESHOLD', () => 128);
//# sourceMappingURL=data:application/json;base64,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