/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
const contexts = {};
const WEBGL_ATTRIBUTES = {
    alpha: false,
    antialias: false,
    premultipliedAlpha: false,
    preserveDrawingBuffer: false,
    depth: false,
    stencil: false,
    failIfMajorPerformanceCaveat: true
};
export function clearWebGLContext(webGLVersion) {
    delete contexts[webGLVersion];
}
export function setWebGLContext(webGLVersion, gl) {
    contexts[webGLVersion] = gl;
}
export function getWebGLContext(webGLVersion, customCanvas) {
    if (!(webGLVersion in contexts) || customCanvas != null) {
        const newCtx = getWebGLRenderingContext(webGLVersion, customCanvas);
        if (newCtx !== null) {
            contexts[webGLVersion] = newCtx;
        }
        else {
            console.log('Could not get context for WebGL version', webGLVersion);
            return null;
        }
    }
    const gl = contexts[webGLVersion];
    if (gl == null || gl.isContextLost()) {
        delete contexts[webGLVersion];
        return getWebGLContext(webGLVersion);
    }
    gl.disable(gl.DEPTH_TEST);
    gl.disable(gl.STENCIL_TEST);
    gl.disable(gl.BLEND);
    gl.disable(gl.DITHER);
    gl.disable(gl.POLYGON_OFFSET_FILL);
    gl.disable(gl.SAMPLE_COVERAGE);
    gl.enable(gl.SCISSOR_TEST);
    gl.enable(gl.CULL_FACE);
    gl.cullFace(gl.BACK);
    return contexts[webGLVersion];
}
function createCanvas(webGLVersion) {
    if (typeof OffscreenCanvas !== 'undefined' && webGLVersion === 2) {
        return new OffscreenCanvas(300, 150);
    }
    else if (typeof document !== 'undefined') {
        return document.createElement('canvas');
    }
    else {
        throw new Error('Cannot create a canvas in this context');
    }
}
function getWebGLRenderingContext(webGLVersion, customCanvas) {
    if (webGLVersion !== 1 && webGLVersion !== 2) {
        throw new Error('Cannot get WebGL rendering context, WebGL is disabled.');
    }
    const canvas = customCanvas == null ? createCanvas(webGLVersion) : customCanvas;
    canvas.addEventListener('webglcontextlost', (ev) => {
        ev.preventDefault();
        delete contexts[webGLVersion];
    }, false);
    if (webGLVersion === 1) {
        return (canvas.getContext('webgl', WEBGL_ATTRIBUTES) ||
            canvas.getContext('experimental-webgl', WEBGL_ATTRIBUTES));
    }
    return canvas.getContext('webgl2', WEBGL_ATTRIBUTES);
}
//# sourceMappingURL=data:application/json;base64,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