/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { Transform, util } from '@tensorflow/tfjs-core';
export function transform(args) {
    const { inputs, attrs, backend } = args;
    const { image, transforms } = inputs;
    const { interpolation, fillMode, fillValue, outputShape } = attrs;
    const [batch, imageHeight, imageWidth, numChannels] = image.shape;
    const [outHeight, outWidth] = outputShape != null ? outputShape : [imageHeight, imageWidth];
    const outShape = [batch, outHeight, outWidth, numChannels];
    const strides = util.computeStrides(image.shape);
    const batchStride = strides[0];
    const rowStride = strides[1];
    const colStride = strides[2];
    const outVals = util.getTypedArrayFromDType(image.dtype, util.sizeFromShape(outShape));
    outVals.fill(fillValue);
    const imageVals = backend.data.get(image.dataId).values;
    const transformVals = backend.data.get(transforms.dataId).values;
    // Ref TF implementation:
    // https://github.com/tensorflow/tensorflow/blob/master/tensorflow/core/kernels/image/image_ops.h
    for (let b = 0; b < batch; ++b) {
        const transform = transforms.shape[0] === 1 ?
            transformVals :
            transformVals.subarray(b * 8, b * 8 + 8);
        for (let outY = 0; outY < outHeight; ++outY) {
            for (let outX = 0; outX < outWidth; ++outX) {
                for (let channel = 0; channel < numChannels; ++channel) {
                    let val;
                    const projection = transform[6] * outX + transform[7] * outY + 1;
                    if (projection === 0) {
                        // Return the fill value for infinite coordinates,
                        // which are outside the input image
                        continue;
                    }
                    const inX = (transform[0] * outX + transform[1] * outY + transform[2]) /
                        projection;
                    const inY = (transform[3] * outX + transform[4] * outY + transform[5]) /
                        projection;
                    const x = mapCoord(inX, imageWidth, fillMode);
                    const y = mapCoord(inY, imageHeight, fillMode);
                    switch (interpolation) {
                        case 'nearest':
                            val = nearestInterpolation(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, b, y, x, channel, fillValue);
                            break;
                        case 'bilinear':
                            val = bilinearInterpolation(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, b, y, x, channel, fillValue);
                            break;
                        default:
                            throw new Error(`Error in Transform: Expect 'nearest' or ` +
                                `'bilinear', but got ${interpolation}`);
                    }
                    const ind = b * batchStride + outY * rowStride + outX * colStride + channel;
                    outVals[ind] = val;
                }
            }
        }
        return backend.makeTensorInfo(outShape, image.dtype, outVals);
    }
    const dataId = backend.write(outVals, outShape, image.dtype);
    return { dataId, shape: image.shape, dtype: image.dtype };
}
export const transformConfig = {
    kernelName: Transform,
    backendName: 'cpu',
    kernelFunc: transform
};
function mapCoord(outCoord, len, mode) {
    switch (mode) {
        case 'reflect':
            return mapCoordReflect(outCoord, len);
        case 'wrap':
            return mapCoordWrap(outCoord, len);
        case 'nearest':
            return mapCoordNearest(outCoord, len);
        case 'constant':
        default:
            return mapCoordConstant(outCoord, len);
    }
}
function mapCoordReflect(outCoord, len) {
    // Reflect [abcd] to [dcba|abcd|dcba].
    let inCoord = outCoord;
    if (inCoord < 0) {
        if (len <= 1) {
            inCoord = 0;
        }
        else {
            const sz2 = 2 * len;
            if (inCoord < sz2) {
                inCoord = sz2 * Math.trunc(-inCoord / sz2) + inCoord;
            }
            inCoord = inCoord < -len ? inCoord + sz2 : -inCoord - 1;
        }
    }
    else if (inCoord > len - 1) {
        if (len <= 1) {
            inCoord = 0;
        }
        else {
            const sz2 = 2 * len;
            inCoord -= sz2 * Math.trunc(inCoord / sz2);
            if (inCoord >= len) {
                inCoord = sz2 - inCoord - 1;
            }
        }
    }
    // clamp is necessary because when outCoord = 3.5 and len = 4,
    // inCoord = 3.5 and will be rounded to 4 in nearest interpolation.
    return util.clamp(0, inCoord, len - 1);
}
function mapCoordWrap(outCoord, len) {
    // Wrap [abcd] to [abcd|abcd|abcd].
    let inCoord = outCoord;
    if (inCoord < 0) {
        if (len <= 1) {
            inCoord = 0;
        }
        else {
            const sz = len - 1;
            inCoord += len * (Math.trunc(-inCoord / sz) + 1);
        }
    }
    else if (inCoord > len - 1) {
        if (len <= 1) {
            inCoord = 0;
        }
        else {
            const sz = len - 1;
            inCoord -= len * Math.trunc(inCoord / sz);
        }
    }
    // clamp is necessary because when outCoord = -0.5 and len = 4,
    // inCoord = 3.5 and will be rounded to 4 in nearest interpolation.
    return util.clamp(0, inCoord, len - 1);
}
function mapCoordConstant(outCoord, len) {
    return outCoord;
}
function mapCoordNearest(outCoord, len) {
    return util.clamp(0, outCoord, len - 1);
}
function readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, y, x, channel, fillValue) {
    const ind = batch * batchStride + y * rowStride + x * colStride + channel;
    if (0 <= y && y < imageHeight && 0 <= x && x < imageWidth) {
        return imageVals[ind];
    }
    else {
        return fillValue;
    }
}
function nearestInterpolation(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, y, x, channel, fillValue) {
    const $y = Math.round(y);
    const $x = Math.round(x);
    return readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, $y, $x, channel, fillValue);
}
function bilinearInterpolation(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, y, x, channel, fillValue) {
    const yFloor = Math.floor(y);
    const xFloor = Math.floor(x);
    const yCeil = yFloor + 1;
    const xCeil = xFloor + 1;
    // f(x, yFloor) = (xCeil - x) / (xCeil - xFloor) * f(xFloor, yFloor)
    //               + (x - xFloor) / (xCeil - xFloor) * f(xCeil, yFloor)
    const valueYFloor = (xCeil - x) *
        readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, yFloor, xFloor, channel, fillValue) +
        (x - xFloor) *
            readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, yFloor, xCeil, channel, fillValue);
    // f(x, yCeil) = (xCeil - x) / (xCeil - xFloor) * f(xFloor, yCeil)
    //             + (x - xFloor) / (xCeil - xFloor) * f(xCeil, yCeil)
    const valueYCeil = (xCeil - x) *
        readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, yCeil, xFloor, channel, fillValue) +
        (x - xFloor) *
            readWithFillValue(imageVals, imageHeight, imageWidth, batchStride, rowStride, colStride, batch, yCeil, xCeil, channel, fillValue);
    // f(x, y) = (yCeil - y) / (yCeil - yFloor) * f(x, yFloor)
    //         + (y - yFloor) / (yCeil - yFloor) * f(x, yCeil)
    return (yCeil - y) * valueYFloor + (y - yFloor) * valueYCeil;
}
//# sourceMappingURL=data:application/json;base64,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