/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
export class IORouterRegistry {
    constructor() {
        this.saveRouters = [];
        this.loadRouters = [];
    }
    static getInstance() {
        if (IORouterRegistry.instance == null) {
            IORouterRegistry.instance = new IORouterRegistry();
        }
        return IORouterRegistry.instance;
    }
    /**
     * Register a save-handler router.
     *
     * @param saveRouter A function that maps a URL-like string onto an instance
     * of `IOHandler` with the `save` method defined or `null`.
     */
    static registerSaveRouter(saveRouter) {
        IORouterRegistry.getInstance().saveRouters.push(saveRouter);
    }
    /**
     * Register a load-handler router.
     *
     * @param loadRouter A function that maps a URL-like string onto an instance
     * of `IOHandler` with the `load` method defined or `null`.
     */
    static registerLoadRouter(loadRouter) {
        IORouterRegistry.getInstance().loadRouters.push(loadRouter);
    }
    /**
     * Look up IOHandler for saving, given a URL-like string.
     *
     * @param url
     * @returns If only one match is found, an instance of IOHandler with the
     * `save` method defined. If no match is found, `null`.
     * @throws Error, if more than one match is found.
     */
    static getSaveHandlers(url) {
        return IORouterRegistry.getHandlers(url, 'save');
    }
    /**
     * Look up IOHandler for loading, given a URL-like string.
     *
     * @param url
     * @param loadOptions Optional, custom load options.
     * @returns All valid handlers for `url`, given the currently registered
     *   handler routers.
     */
    static getLoadHandlers(url, loadOptions) {
        return IORouterRegistry.getHandlers(url, 'load', loadOptions);
    }
    static getHandlers(url, handlerType, loadOptions) {
        const validHandlers = [];
        const routers = handlerType === 'load' ?
            IORouterRegistry.getInstance().loadRouters :
            IORouterRegistry.getInstance().saveRouters;
        routers.forEach(router => {
            const handler = router(url, loadOptions);
            if (handler !== null) {
                validHandlers.push(handler);
            }
        });
        return validHandlers;
    }
}
export const registerSaveRouter = (loudRouter) => IORouterRegistry.registerSaveRouter(loudRouter);
export const registerLoadRouter = (loudRouter) => IORouterRegistry.registerLoadRouter(loudRouter);
export const getSaveHandlers = (url) => IORouterRegistry.getSaveHandlers(url);
export const getLoadHandlers = (url, loadOptions) => IORouterRegistry.getLoadHandlers(url, loadOptions);
//# sourceMappingURL=router_registry.js.map